#####################################################################################################
## Palate Fusion Model - STEPPABLES                                                                ##
##                                                                                                 ##
## National Center for Computational Toxicology                                                    ##
## United States Environmental Protection Agency                                                   ##
## http://www.epa.gov/ncct/v-embryo/                                                               ##
##                                                                                                 ##
## Author:  M. Shane Hutson                                                                        ##
## Initial Date:  09-17-2012                                                                       ##
## Revision Date: 08-15-2015                                                                       ##
##                                                                                                 ##
## CC3D Version: 3.7.3                                                                             ##
#####################################################################################################

from PySteppables import *
from PySteppablesExamples import MitosisSteppableClustersBase
from CompuCell import NeighborFinderParams
import CompuCell
import sys
from PlayerPython import *
from math import *
from CompuCell import Point3D
import random
from XMLUtils import dictionaryToMapStrStr as d2mss
import types
from timeit import Timer
import time
from numpy.random import binomial 
import numpy

#****************************************************************************************************
# Define constants for referencing Cell Types
MEDIUM = 0
WALL = 1
MESENCH_N = 2
MESENCH_O = 3
MESENCH_EMT = 4
MATRIX = 5
EPI_N = 6
EPI_O = 7
EPI_ME = 8
BM = 9
PERI_N = 10
PERI_O = 11
PERI_ME = 12
PERI_Na = 13
PERI_Oa = 14
PERI_MEa = 15
TJ = 16
TJm = 17
#*****************************************************************************************************
# Additional global variables
nCellTypes = 18 #includes 'Medium' so all lists can be zero indexed
typeList = ["Medium",
            "Wall",
            "Mesench_N",
            "Mesench_O",
            "Mesench_EMT",
            "Matrix",
            "Epi_N",
            "Epi_O",
            "Epi_ME",
            "BaseMemb",
            "Peri_N",
            "Peri_O",
            "Peri_ME",
            "Peri_Na",
            "Peri_Oa",
            "Peri_MEa",
            "TJ",
            "TJm"]
useCellParameterChangeList = False
MITOSIS_VOLUME_LIST=[0 for i in range(nCellTypes)] #This will be modified by initial setup routine
MITOSIS_COUNT_LIST= [0 for i in range(nCellTypes)]
SIMULATION_TIME=time.clock()
SIMULATION_TIME0=time.clock()
DEFAULT_CELL_ATTRIBUTE_DICT = {
    'cell.generation':0,
    'cell.parent':0,
    'cell.parentClusterId':0,
    'cell.needsClusterReassignmentFlag':False,
    'cell.apoptosisFlag':False,
    'cell.mitosisFlag':False,
    'cell.timer.ectoDivision':0,
    'cell.timer.apoptosis':0,
    'cell.timer.mitosis':0,
    'cell.EGFR':0}

## Scalings for Hexagonal lattice
LMFLENGTH_HEX=1*sqrt(2.0/(3.0*sqrt(3.0)))*sqrt(3.0)
XSCALE_HEX=1.0
YSCALE_HEX=1.0*sqrt(3.0)/2.0
ZSCALE_HEX=1.0*sqrt(6.0)/3.0
## Scalings for Hexagonal lattice
#*****************************************************************************************************


#*************************************************************************************************************
#************* EXAMPLE OF BASE ELEMENTS NEEDED IN A STEPPABLE (good start for new steppables) ****************
#*************************************************************************************************************
class baseSteppable(SteppablePy):
   def __init__(self,_simulator,_frequency=1):
      SteppablePy.__init__(self,_frequency)
      self.simulator=_simulator
      self.inventory=self.simulator.getPotts().getCellInventory()
      self.cellList=CellList(self.inventory)
      
   def start(self): pass
   
   def step(self,mcs): pass

#*************************************************************************************************************
class PalateModelCellAttributes(SteppableBasePy):            # Attaches attributes to each cell in the cellList
    def __init__(self,_simulator,_frequency=1):
        SteppableBasePy.__init__(self,_simulator,_frequency)
    
    def start(self):
        [CompuCell.getPyAttrib(cell).update(DEFAULT_CELL_ATTRIBUTE_DICT) for cell in self.cellList]
        for cellType in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_ME,PERI_O,PERI_Na,PERI_MEa,PERI_Oa]:
            [CompuCell.getPyAttrib(cell).update({'cell.EGFR':1}) for cell in self.cellListByType(cellType)]

        for cellType in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_ME,PERI_O]:
            [CompuCell.getPyAttrib(cell).update({'cell.timer.ectoDivision':random.randint(1,101)}) for cell in self.cellListByType(cellType)]
        #[CompuCell.getPyAttrib(cell).update({'cell.timer.ectoDivision':random.randint(1,101)}) for cell in self.cellListByType(EPI_N)]
        #[CompuCell.getPyAttrib(cell).update({'cell.timer.ectoDivision':random.randint(1,101)}) for cell in self.cellListByType(EPI_ME)]
        #[CompuCell.getPyAttrib(cell).update({'cell.timer.ectoDivision':random.randint(1,101)}) for cell in self.cellListByType(PERI_N)]
        #[CompuCell.getPyAttrib(cell).update({'cell.timer.ectoDivision':random.randint(1,101)}) for cell in self.cellListByType(PERI_ME)]

    def step(self,mcs):
        x=time.clock()
        print "\n***** PalateModelCellAttributes (step) ***************************"
        EGFR_OK_flag = True
        for cell in self.cellList:
            cellAttributes=CompuCell.getPyAttrib(cell)
            cellAttributes['cell.timer.mitosis'] += self.frequency
            if cellAttributes['cell.timer.ectoDivision']>0: cellAttributes['cell.timer.ectoDivision']-=self.frequency
            #print "cell.EGFR = ",cellAttributes['cell.EGFR']
            if cell.type in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_ME,PERI_O,PERI_Na,PERI_MEa,PERI_Oa]: 
                EGFR_OK_flag = EGFR_OK_flag and cellAttributes['cell.EGFR']==1
            else:
                EGFR_OK_flag = EGFR_OK_flag and cellAttributes['cell.EGFR']==0
        if EGFR_OK_flag:
            print "EGFR = 1 for all epithelial cells and EGFR = 0 for non-epithelial cells"
        else:
            print "PROBLEM: EGFR != 1 for all epithelial cells or EGFR != 0 for all non-epithelial cells"    
        print "PalateModelCellAttributes,   Time Taken=", time.clock()-x
        

#*************************************************************************************************************
class SetInitial2DGeometry(SteppablePy):
   def __init__(self,_simulator,_frequency=1):
    print "\n***** SetInitial2DGeometry Steppable (__init__)"
    SteppablePy.__init__(self,_frequency)
    self.runBeforeMCS=1   # needed to make sure this runs before the "behavior" steppable that sets target volumes
    self.simulator=_simulator
    self.inventory=self.simulator.getPotts().getCellInventory()
    self.cellList=CellList(self.inventory)
    self.Potts = self.simulator.getPotts()
    self.cellFieldG = self.Potts.getCellFieldG()
    self.dim=self.cellFieldG.getDim()
    #default values
    self.includePolarizedLeadingEdge=False
    self.setSpecialMedialEdge=False
    self.includeMirrorShelf=False
    self.useHexLattice=True
    self.tissueButtonList=[]
    
   def specifyTissueButton(self, _x0, _y0, _rMesench, _drEpi, _drPeri, _polDirection, _nCellsList, _cellTypeList):
    circleParamMesench = {'nCells':_nCellsList[0], 'x0':_x0, 'y0':_y0, 
                          'r1':0,                'r2':_rMesench,                'theta':_polDirection, 
                          'cellTypeList':_cellTypeList[0]}
    circleParamEpi =     {'nCells':_nCellsList[1], 'x0':_x0, 'y0':_y0, 
                          'r1':_rMesench,        'r2':_rMesench+_drEpi,         'theta':_polDirection, 
                          'cellTypeList':_cellTypeList[1]}
    circleParamPeri =    {'nCells':_nCellsList[2],'x0':_x0, 'y0':_y0, 
                          'r1':_rMesench+_drEpi, 'r2':_rMesench+_drEpi+_drPeri, 'theta':_polDirection, 
                          'cellTypeList':_cellTypeList[2]}
    self.tissueButtonList.append(circleParamMesench)
    self.tissueButtonList.append(circleParamEpi)
    self.tissueButtonList.append(circleParamPeri)
    return

   def createCell(self,_pt,_type,_clusterId):
    # print "\n***** SetInitial2DGeometry Steppable (createCell)"
    print "Creating cell #",_clusterId," at pt ",str(_pt)," of type ",typeList[_type]
    cell=self.Potts.createCellG(_pt)
    cell.type=_type
    # deprecated direct assignment of clusterId
    # cell.clusterId=_clusterId
    self.inventory.reassignClusterId(cell,_clusterId)
    print "\t cell created successfully"
    return cell

   def getCentroid(self, _cell, _hexLattice):
        if _cell.volume>0 and _hexLattice:
            x=int(_cell.xCM/(float(_cell.volume)*XSCALE_HEX))
            y=int(_cell.yCM/(float(_cell.volume)*YSCALE_HEX)) 
            z=int(_cell.zCM/(float(_cell.volume)*ZSCALE_HEX))
            return Point3D(x,y,z)
        elif _cell.volume>0:
            x=int(_cell.xCM/(float(_cell.volume)))
            y=int(_cell.yCM/(float(_cell.volume))) 
            z=int(_cell.zCM/(float(_cell.volume)))
            return Point3D(x,y,z)
        else:
            return Point3D(0,0,0)

   def getDistance2(self, _pt1, _pt2):
    return (_pt1.x-_pt2.x)**2 \
         + (_pt1.y-_pt2.y)**2 \
         + (_pt1.z-_pt2.z)**2
                         
   def getDistance(self, _pt1, _pt2):
    return sqrt(self.getDistance2(_pt1,_pt2))
    
   def getDistance2FromList(self, _coordList1, _coordList2):
    if len(_coordList1) != len(_coordList2) or len(_coordList1)<2: 
        print "!!!!ERROR: Dimensional mismatch in getDistance"
        return
    d2 = ((_coordList1[0]-_coordList2[0]))**2 \
       + ((_coordList1[1]-_coordList2[1]))**2
    if len(_coordList1)>2:
        d2 += ((_coordList1[2]-_coordList2[2]))**2    
    return d2
    
   def getDistanceFromList(self, _coordList1, _coordList2):
    return sqrt(self.getDistance2FromList(_coordList1,_coordList2))
    
   def getVectorAngle(self, _pt1, _pt2): # from _pt1 to _pt2
    return atan2( (_pt2.y-_pt1.y), (_pt2.x-_pt1.x))
    
   def getClosestCell(self,_pt,_allowedTypeList):
    mindist2 = self.getDistance2(Point3D(0,0,0),Point3D(self.dim.x,self.dim.y,self.dim.z))
    closestcell=self.cellFieldG.get(_pt)
    for cell in self.cellList:
        if cell and (cell.type in _allowedTypeList) and cell.volume>0:
            #ptC = Point3D(int(cell.xCM/float(cell.volume)),int(cell.yCM/float(cell.volume)),int(cell.zCM/float(cell.volume)))
            ptC = self.getCentroid(cell, _hexLattice=self.useHexLattice)
            dist2 = self.getDistance2(_pt,ptC)
            if dist2<mindist2:
                mindist2 = dist2
                closestcell=cell
    return closestcell
    
   def getPixelListBetweenCircles(self, _x0, _y0, _r1, _r2 ):
    matchingPixelList=[]
    for x in range(1,int(self.dim.x)-1):
        for y in range(1,int(self.dim.y)-1):
            r = self.getDistanceFromList([x0,y0],[x,y])
            if _r1 <= r <= _r2:
                matchingPixelList.append(Point3D(x,y,0))
    return matchingPixelList

   def getMultiPixelListBetweenCircles(self, _circleParamList ):
        matchingPixelListList=[ [] for entry in _circleParamList]
        for x in range(1,int(self.dim.x)-1):
            for y in range(1,int(self.dim.y)-1):
                for i, circleParam in enumerate(_circleParamList):
                    r = self.getDistanceFromList([circleParam["x0"],circleParam["y0"]],[x,y])
                    if circleParam["r1"] <= r < circleParam["r2"]:
                        matchingPixelListList[i].append(Point3D(x,y,0))
        return matchingPixelListList


   def SetIncludePolarizedLeadingEdge(self, _includePolarizedLeadingEdge):
    print "\n***** SetInitial2DGeometry Steppable (SetIncludePolarizedLeadingEdge)"
    self.includePolarizedLeadingEdge = _includePolarizedLeadingEdge
    
   def SetSpecialMedialEdge(self, _includeSpecialMedialEdge):
    print "\n***** SetInitial2DGeometry Steppable (SetSpecialMedialEdge)"
    self.setSpecialMedialEdge = _includeSpecialMedialEdge

   def IncludeMirrorShelf(self, _includeMirrorShelf):
    print "\n***** SetInitial2DGeometry Steppable (IncludeMirrorShelf)"
    self.includeMirrorShelf = _includeMirrorShelf

   def createWall(self, _index):
    print "Creating WALL cell"
    verbose = False
    pt = Point3D(0,0,0)
    wallcell = self.createCell(pt,WALL,_index)
    for x in range(0,int(self.dim.x)):
        pt1 = Point3D(int(x),0,0)
        pt2 = Point3D(int(x),int(self.dim.y-1),0)
        self.cellFieldG.set(pt1,wallcell)
        if verbose: print "added point to wallcell at ",str(pt1)
        self.cellFieldG.set(pt2,wallcell)
        if verbose: print "added point to wallcell at ",str(pt2)
    for y in range(0,int(self.dim.y)):
        pt1 = Point3D(0,int(y),0)
        pt2 = Point3D(int(self.dim.x-1),int(y),0)
        self.cellFieldG.set(pt1,wallcell)
        if verbose: print "added point to wallcell at ",str(pt1)
        self.cellFieldG.set(pt2,wallcell)
        if verbose: print "added point to wallcell at ",str(pt2)
    return
        
   def start(self):
    print "\n***** SetInitial2DGeometry Steppable (start)"
    #first define the WALL cell as a border around entire lattice
    index = 1
    self.createWall(index)
    index+=1
    #now add mesenchymal, epithelial and periderm cells
    #according to user-specified geometry of initial tissue buttons
    pixelListList = self.getMultiPixelListBetweenCircles(self.tissueButtonList)
       
    seedListList=[random.sample(pixelListList[i],tissueButton['nCells']) for i,tissueButton in enumerate(self.tissueButtonList)]
    for i,seedList in enumerate(seedListList):
        cellType1 = (self.tissueButtonList[i]['cellTypeList'])[0]
        # check to see if this part of the tissue should be polarized into two or three sections
        if (self.tissueButtonList[i]).has_key('theta') and len(self.tissueButtonList[i]['cellTypeList'])>1:
            theta = self.tissueButtonList[i]['theta']
            x0 = self.tissueButtonList[i]['x0']
            y0 = self.tissueButtonList[i]['y0']
            cellType2 = (self.tissueButtonList[i]['cellTypeList'])[1]
            # polarization into three sections
            if len(self.tissueButtonList[i]['cellTypeList'])>2:
                print "Polarization in THREE Compartments"
                cellType3 = (self.tissueButtonList[i]['cellTypeList'])[2]
                for pt in seedList:
                    # check if seed point is within some range (+/- pi/6) of theta direction
                    phi = self.getVectorAngle(Point3D(int(x0),int(y0),0),pt)
                    #phi = atan2(pt.y-y0,pt.x-x0)
                    if cos(phi-theta)<0.866: #same as checking whether angles outside +/- 30 degrees
                        if sin(phi-theta)>=0: newcell = self.createCell(pt, cellType1, index)
                        else:                 newcell = self.createCell(pt, cellType2, index)
                    else:
                        newcell = self.createCell(pt, cellType3, index)
                    index+=1
            # polarization into two sections
            else:
                print "Polarization in TWO Compartments"
                for pt in seedList:
                    # check if seed point is above or below theta direction
                    phi = self.getVectorAngle(Point3D(int(x0),int(y0),0),pt)
                    #phi = atan2(pt.y-y0,pt.x-x0)
                    if sin(phi-theta)>=0: newcell = self.createCell(pt, cellType1, index)
                    else:                 newcell = self.createCell(pt, cellType2, index)
                    index+=1
        else:
            print "No polarization ONE Compartment"
            for pt in seedList:
                newcell = self.createCell(pt, cellType1, index)
                index+=1
    print "Found all seed points for Voronoi tesselation"

    for i, pixelList in enumerate(pixelListList):
        relevantCellTypes = self.tissueButtonList[i]['cellTypeList']
        print "Assigning pixels to cells for region",i+1,"of",len(pixelListList)
        npts = len(pixelList)
        for i,pt in enumerate(pixelList):
            if (i+1)%(int(0.1*npts))==0: print "\t",100*(i+1)/npts,"% complete"
            cellAtPoint3D=self.cellFieldG.get(pt)
            if not cellAtPoint3D:
                closestCell = self.getClosestCell(pt,relevantCellTypes)
                self.cellFieldG.set(pt, closestCell)
                                
   def step(self,mcs): pass

           
#*************************************************************************************************************
class PalateModelCellBehaviors(SteppableBasePy):     #Implements cell growth, can be later based on chemical fields
    def __init__(self,_simulator,_frequency=1):
        SteppableBasePy.__init__(self,_simulator,_frequency)
        self.runBeforeMCS=1 # needed to make sure SecretionPlugin works correctly, i.e. secretion before diffusion solver
        self.Potts = self.simulator.getPotts()
        self.cellFieldG = self.Potts.getCellFieldG()
        # default values for options
        self.useHexLattice = True
        self.useAvgVolumeForEachCellType = False
        self.useInitTargetSurfaceBasedOnVolume = False
        self.useRandomVolumeForMesenchCells = False
        self.reduceMotilityOfPeriCells = True
        self.mcsAtWhichToDecreasePeriCellMotility = 1000000000  # essentially never by default
        self.mcsAtWhichToIncreasePeriCellMotility = 1000000000  # essentially never by default
        self.timeToStopGrowth = 3200   # default value
        self.useMatrix = True
        self.useBM = True  # whether an explicit basement membrane will be used
        self.useAspectRatioForMitosis = False
        self.apoptosisTimeConstant = 200
        self.allowApoptosisTime = 500
        self.allowEmtTime = 500
        self.TGFbetaFlag = True
        self.doNotDivideTypeList = [MEDIUM,WALL,MATRIX,BM,TJ,TJm]
        self.clusterBaseTypeList = [MESENCH_N,MESENCH_O,MESENCH_EMT,EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME,MATRIX,BM]
        self.geneStatusDict = {
            'TGFb3':True,
            'FGF10':True,
            'FGF7':True,
            'BMP2':True,
            'BMP4':True,
            'NOGGIN':True,
            'SHH':True,
            'MMP':True}  # set a value to False to mimic genetic knockout of the key (gene name)
        self.ahrActivity = 0.0
        self.ahrDrivenEGFRMaxFoldChange = 1.0  # 1.0 is no change in EGFR levels
        self.ahrDrivenEGFRStartTime = 1000
        self.ahrDrivenEGFRTauUp = 100
        self.ahrDrivenEGFRTauDown = 600
        self.EGFR_0 = 130
        self.TGFBR_0 = 150
        self.useMatrixElasticity = False
        self.fppLambdaMax = 100
        self.fppTimeConstant = 100
        self.verbose = False
    
    def getAhrDrivenEGFRFoldChange(self, _mcs):
        foldChange = 1.0 # default until mcs reaches start time
        if _mcs > self.ahrDrivenEGFRStartTime:
            t0 = self.ahrDrivenEGFRStartTime
            tauUp = self.ahrDrivenEGFRTauUp
            tauDown = self.ahrDrivenEGFRTauDown
            deltaFmax = self.ahrDrivenEGFRMaxFoldChange - 1.0    
            factorUp = 1.0 - exp(-1.0*(_mcs-t0)/tauUp)
            factorDown = exp(-1.0*(_mcs-t0)/tauDown)
            foldChange = 1.0 + deltaFmax*factorUp*factorDown
        return foldChange
        
    def initTargetVolumeToAvgForEachCellType(self,_useAvgVolumeForEachCellType):
        self.useAvgVolumeForEachCellType=_useAvgVolumeForEachCellType
    
    def initTargetSurfaceBasedOnVolume(self,_useInitTargetSurfaceBasedOnVolume):
        self.useInitTargetSurfaceBasedOnVolume=_useInitTargetSurfaceBasedOnVolume
    
    def getBoundaryCell(self, _cell, _boundedCellTypeList):
        cellNeighborList=CellNeighborListAuto(self.neighborTrackerPlugin,_cell)
        boundaryCellFlag = False
        matchingNeighborsList=[]
        for neighborSurfaceData in cellNeighborList:
            if neighborSurfaceData.neighborAddress:
                if neighborSurfaceData.neighborAddress.type in _boundedCellTypeList:
                    boundaryCellFlag=True
                    matchingNeighborsList.append(neighborSurfaceData.neighborAddress)
            elif 0 in _boundedCellTypeList: #necessary to consider cells that border Medium,
                # only gets to this elif when neighborSurfaceData.neighborAddress doesn't exist, i.e. neighbor is Medium
                boundaryCellFlag = True
        if boundaryCellFlag: return random.choice(matchingNeighborsList)
        else: return boundaryCellFlag       # return a random matching cell if one found, return False if not found
    
    def isBoundaryCell(self, _cell, _boundedCellTypeList):
        cellNeighborList=CellNeighborListAuto(self.neighborTrackerPlugin,_cell)
        boundaryCellFlag = False
        for neighborSurfaceData in cellNeighborList:
            if neighborSurfaceData.neighborAddress:
                if neighborSurfaceData.neighborAddress.type in _boundedCellTypeList:
                    boundaryCellFlag=True
            elif 0 in _boundedCellTypeList: #necessary to consider cells that border Medium,
                # only gets to this elif when neighborSurfaceData.neighborAddress doesn't exist, i.e. neighbor is Medium
                boundaryCellFlag = True
        return boundaryCellFlag       

    def getNeighborCellTypes(self, _cell):
        cellNeighborList=CellNeighborListAuto(self.neighborTrackerPlugin,_cell)
        neighborCellTypeList=[]
        for neighborSurfaceData in cellNeighborList:
            if neighborSurfaceData.neighborAddress:
                neighborCellTypeList.append(neighborSurfaceData.neighborAddress.type)
            else:
                neighborCellTypeList.append(0)  #needed to include MEDIUM as a neighboring cell type 
        return neighborCellTypeList

    def calculateAvgPropertiesByType(self):
        localAvgTypeVolume=[0 for x in range(nCellTypes)]
        localAvgTypeSurface=[0 for x in range(nCellTypes)]
        localNumType=[0 for x in range(nCellTypes)]
        for cell in self.cellList:
            if cell:
                localNumType[cell.type] += 1
                localAvgTypeVolume[cell.type] += cell.volume
                localAvgTypeSurface[cell.type] += cell.surface
        print "\nAVERAGE VOLUME AND SURFACE AREA OF EACH CELL TYPE"
        print "CellType\tAverageVolume\tAverageSurfaceArea"
        print "---------------------------------------------------------------"
        for i in range(nCellTypes):
            if localNumType[i]>0:
                localAvgTypeVolume[i] = localAvgTypeVolume[i]/localNumType[i]
                localAvgTypeSurface[i] = localAvgTypeSurface[i]/localNumType[i]
                print "%(type)9s\t%(avgVol)9.1f\t%(avgSurf)9.1f" % {
		      'type':typeList[i], 'avgVol':localAvgTypeVolume[i], 'avgSurf':localAvgTypeSurface[i]
		    }
	return

    def nucleateMatrixCell(self, _cell):
        if self.verbose: print "Entering nucleateMatrixCell function"
        if _cell.volume>2 and _cell.targetVolume>2:
            pixelTrackerDataList=CellBoundaryPixelList(self.boundaryPixelTrackerPlugin,_cell)    #careful, not really a python list
            pixelList=[]
            for boundaryPixelTrackerData in pixelTrackerDataList: pixelList.append(boundaryPixelTrackerData.pixel)
            pt = random.choice(pixelList)
            if pt:
                if self.verbose: print "Nucleating MATRIX from cell.id=",_cell.id,"of type",typeList[_cell.type],"at point=",pt
                newMatrix=self.Potts.createCellG(pt)
                newMatrix.type=MATRIX
                newMatrix.targetVolume = 1
                newMatrix.lambdaVolume = 5*_cell.lambdaVolume
                CompuCell.getPyAttrib(newMatrix).update(DEFAULT_CELL_ATTRIBUTE_DICT)
                return newMatrix
            else:
                if self.verbose: print "No MATRIX nucleated; could not find appropriate pt"
                return  # can use this to assess whether MATRIX was actually nucleated
        if self.verbose: print "No MATRIX nucleated; cell.volume >= 2"
        return # can use this to assess whether MATRIX was actually nucleated

    def nucleateBM(self, _cell):
       if True: print "Entering nucleateBM function"
       if _cell.volume>4 and _cell.targetVolume>4:
           #pt = self.pickSurfacePixel(_cell,[MESENCH_N,MESENCH_O,MESENCH_EMT,MATRIX], False) #random point on surface
           [pt1, pt2] = self.pickTwoSurfacePixels(_cell,[MESENCH_N,MESENCH_O,MESENCH_EMT,MATRIX], False) #two random adjacent points on surface
           if pt1 and pt2:
               if True: print "Nucleating BMs from cell.id=",_cell.id,"of type",typeList[_cell.type],"at points =",pt1," and ",pt2
               newBM1=self.Potts.createCellG(pt1)
               newBM2=self.Potts.createCellG(pt2)
               newBM1.type=BM
               newBM2.type=BM
               newBM1.targetVolume = 3.5
               newBM2.targetVolume = 3.5
               newBM1.lambdaVolume = 40.0*_cell.lambdaVolume
               newBM2.lambdaVolume = 40.0*_cell.lambdaVolume
               # need to change surface and length constraints too; goal is "cells" that are 1x4 pixels
                  #newBM.targetSurface = newBM.targetVolume
                  #newBM.lambdaSurface = 1.0*newBM.lambdaVolume
               #self.lengthConstraintPlugin.setLengthConstraintData(newBM, newBM.lambdaVolume, newBM.targetVolume)
               CompuCell.getPyAttrib(newBM1).update(DEFAULT_CELL_ATTRIBUTE_DICT)
               CompuCell.getPyAttrib(newBM2).update(DEFAULT_CELL_ATTRIBUTE_DICT)
               # reassign clusterId of newBM2 "cell" to match that of newBM1 "cell"
               self.inventory.reassignClusterId(newBM2, newBM1.clusterId)
               return newBM1
           else:
               if True: print "No BM nucleated; could not find appropriate pts"
               return  # can use this to assess whether BM was actually nucleated
       if True: print "No BM nucleated; cell.volume <= 4"
       return # can use this to assess whether BM was actually nucleated

    def pickSurfacePixel(self, _cell, _boundedCellTypeList, _awayFromPoint):
        if self.verbose: print "Trying to find pt in cell",_cell.id,"that bounds cell of type",_boundedCellTypeList
        pixelList=CellBoundaryPixelList(self.boundaryPixelTrackerPlugin,_cell)
        selectPixelList=[]
        for boundaryPixelTrackerData in pixelList:
            addThisPixel=False
            #if self.verbose: print "pixel of cell id=",_cell.id," type:",_cell.type, " = ",boundaryPixelTrackerData.pixel
            # check which pixels border cell types in _boundedCellTypeList
            pt = boundaryPixelTrackerData.pixel
            pt1 = Point3D(pt.x+1, pt.y, pt.z)
            pt2 = Point3D(pt.x, pt.y+1, pt.z)
            pt3 = Point3D(pt.x-1, pt.y, pt.z)
            pt4 = Point3D(pt.x, pt.y-1, pt.z)
            if pt.x==0:            pt3 = Point3D(self.dim.x-1, pt.y, pt.z)
            if pt.x==self.dim.x-1: pt1 = Point3D(0, pt.y, pt.z)
            if pt.y==0:            pt4 = Point3D(pt.x, self.dim.y-1, pt.z)
            if pt.y==self.dim.y-1: pt2 = Point3D(pt.x, 0, pt.z)
            cell1 = self.cellFieldG.get(pt1)
            cell2 = self.cellFieldG.get(pt2)
            cell3 = self.cellFieldG.get(pt3)
            cell4 = self.cellFieldG.get(pt4)
            for cell in [cell1, cell2, cell3, cell4]:
                if 0 in _boundedCellTypeList and not cell: addThisPixel=True
                elif cell and cell.type in _boundedCellTypeList and cell.clusterId!=_cell.clusterId: addThisPixel=True
            if addThisPixel:
                selectPixelList.append(pt)
                #if self.verbose: print "         at surface with cell.types=",_boundedCellTypeList
        if len(selectPixelList)>=1:
            if not _awayFromPoint:
               return random.choice(selectPixelList) #selectPixelList[random.randint(0,len(selectPixelList)-1)]
            else:
                farthestPoint=selectPixelList[0]
                maxDsquared=0
                for pt in selectPixelList:
                    dsquared =(pt.x - _awayFromPoint.x)**2+(pt.y - _awayFromPoint.y)**2+(pt.z - _awayFromPoint.z)**2
                    if dsquared>maxDsquared:
                        maxDsquared=dsquared
                        farthestPoint=pt
                return farthestPoint
        else: return

    def pickTwoSurfacePixels(self, _cell, _boundedCellTypeList, _awayFromPoint):
        if True: print "Trying to find two pts in cell",_cell.id,"that bound cells of type",_boundedCellTypeList
        pixelList=CellBoundaryPixelList(self.boundaryPixelTrackerPlugin,_cell)
        selectPixelList=[]
        for boundaryPixelTrackerData in pixelList:
           addThisPixel=False
           #if self.verbose: print "pixel of cell id=",_cell.id," type:",_cell.type, " = ",boundaryPixelTrackerData.pixel
           # check which pixels border cell types in _boundedCellTypeList
           pt = boundaryPixelTrackerData.pixel
           pt1 = Point3D(pt.x+1, pt.y, pt.z)
           pt2 = Point3D(pt.x, pt.y+1, pt.z)
           pt3 = Point3D(pt.x-1, pt.y, pt.z)
           pt4 = Point3D(pt.x, pt.y-1, pt.z)
           if pt.x==0:            pt3 = Point3D(self.dim.x-1, pt.y, pt.z)
           if pt.x==self.dim.x-1: pt1 = Point3D(0, pt.y, pt.z)
           if pt.y==0:            pt4 = Point3D(pt.x, self.dim.y-1, pt.z)
           if pt.y==self.dim.y-1: pt2 = Point3D(pt.x, 0, pt.z)
           cell1 = self.cellFieldG.get(pt1)
           cell2 = self.cellFieldG.get(pt2)
           cell3 = self.cellFieldG.get(pt3)
           cell4 = self.cellFieldG.get(pt4)
           for cell in [cell1, cell2, cell3, cell4]:
              if 0 in _boundedCellTypeList and not cell: addThisPixel=True
              elif cell and cell.type in _boundedCellTypeList and cell.clusterId!=_cell.clusterId: addThisPixel=True
           if addThisPixel:
              selectPixelList.append(pt)
        #if self.verbose: print "         at surface with cell.types=",_boundedCellTypeList
        if len(selectPixelList)>=2:
           pt1 = random.choice(selectPixelList) #selectPixelList[random.randint(0,len(selectPixelList)-1)]
           if _awayFromPoint:
              farthestPoint=selectPixelList[0]
              maxDsquared=0
              for pt in selectPixelList:
                 dsquared =(pt.x - _awayFromPoint.x)**2+(pt.y - _awayFromPoint.y)**2+(pt.z - _awayFromPoint.z)**2
                 if dsquared>maxDsquared:
                    maxDsquared=dsquared
                    farthestPoint=pt
              pt1 = farthestPoint
           # need to select pt2 as point in selectPixelList that is closest to pt1
           adjPixelList=[]
           addThisPixel = False
           for pt in selectPixelList:
              dx = abs(pt.x - pt1.x)
              dy = abs(pt.y - pt1.y)
              dz = abs(pt.z - pt1.z)
              dsquared =(pt.x - pt1.x)**2+(pt.y - pt1.y)**2+(pt.z - pt1.z)**2
              print "Checking pt1 ",pt1," against pt ",pt,"; d**2 = ",dsquared
              #if (dx==1 or dy==1 or dz==1): adjPixelList.append(pt)
              if (dsquared>=1 and dsquared <3): adjPixelList.append(pt)
           if len(adjPixelList)>=1:
              pt2 = random.choice(adjPixelList)
              if True: print "pt1 = ",pt1,", pt2 = ",pt2
              return [pt1, pt2]
           else:
              print "pt1 = ",pt1,", but NO adj pixel in BoundaryPixelList that also neighbored cell of type ",_boundedCellTypeList
              return [pt1, False]
        else:
           if True: print "BoundaryPixelList had too few pixels neighboring cells of type",_boundedCellTypeList
           return [False, False]


    def hillFunction(self, _fieldValue, _maxRate, _AC50, _n=1):
        return _maxRate*((_fieldValue**_n)/(_AC50**_n+_fieldValue**_n))
    
    def getCentroid(self, _cell, _hexLattice):
        if _cell.volume>0 and _hexLattice:
            x=int(_cell.xCM/(float(_cell.volume)*XSCALE_HEX))
            y=int(_cell.yCM/(float(_cell.volume)*YSCALE_HEX)) 
            z=int(_cell.zCM/(float(_cell.volume)*ZSCALE_HEX))
            return Point3D(x,y,z)
        elif _cell.volume>0:
            x=int(_cell.xCM/(float(_cell.volume)))
            y=int(_cell.yCM/(float(_cell.volume))) 
            z=int(_cell.zCM/(float(_cell.volume)))
            return Point3D(x,y,z)
        else:
            return Point3D(0,0,0)
    
    def printSummary(self, _listOfLists, _fStr="%.2f"):
        if len(_listOfLists) != nCellTypes: 
            print "List Length Mismatch"
            return
        else:
            for i in range(nCellTypes):
                if len(_listOfLists[i])>0:
                    summary_str = "\t%11s: "+_fStr+" +/- "+_fStr+" (range = "+_fStr+" to "+_fStr+")"
                    avg = numpy.average(_listOfLists[i])
                    stdev = numpy.std(_listOfLists[i])
                    print summary_str % (typeList[i],avg,stdev,min(_listOfLists[i]),max(_listOfLists[i]))
                
    def getFppLinkDistance(self, _cell, _fppd):   #assumes non-periodic boundaries
        neighbor = _fppd.neighborAddress
        if _cell.volume>0 and neighbor.volume>0:
            pt1 = self.getCentroid(_cell, _hexLattice=self.useHexLattice)
            pt2 = self.getCentroid(neighbor, _hexLattice=self.useHexLattice)
            dx = pt1.x-pt2.x
            dy = pt1.y-pt2.y
            dz = pt1.z-pt2.z
            return sqrt(dx*dx + dy*dy + dz*dz)
        else:
            return 0.0
         
    def start(self):
        # ************* TEST ADDING A PLOT WINDOW ************************
        #self.pW = self.addNewPlotWindow(_title='S1 concentration',_xAxisTitle='MonteCarlo Step (MCS)',_yAxisTitle='Variables')
        #self.pW.addPlot('S1',_style='Dots',_color='red',_size=5)   
        # ************* TEST ADDING A PLOT WINDOW ************************

        print "\n***** PalateModelCellBehaviors Steppable (start)"
        global MITOSIS_VOLUME_LIST 
        verbose = False
        pfactor = 1.0
        LV = 0.5
        LS = 0.1
        #first calculate all the averages, make these variables available to other parts of this steppable
        self.avgTypeVolume=[0 for x in range(nCellTypes)]
        self.avgTypeSurface=[0 for x in range(nCellTypes)]
        self.numType=[0 for x in range(nCellTypes)]
        for cell in self.cellList:
            if cell:
                self.numType[cell.type] += 1
                self.avgTypeVolume[cell.type] += cell.volume
                self.avgTypeSurface[cell.type] += cell.surface
        print "\nAVERAGE VOLUME AND SURFACE AREA OF EACH CELL TYPE"
        print "CellType\tAverageVolume\tAverageSurfaceArea"
        print "---------------------------------------------------------------"
        for i in range(nCellTypes):
            if self.numType[i]>0:
                self.avgTypeVolume[i] = self.avgTypeVolume[i]/self.numType[i]
                self.avgTypeSurface[i] = self.avgTypeSurface[i]/self.numType[i]
                print "%(type)9s\t%(avgVol)9.1f\t%(avgSurf)9.1f" % {'type':typeList[i], 'avgVol':self.avgTypeVolume[i], 'avgSurf':self.avgTypeSurface[i]}
        for i in range(nCellTypes):
            #set volumes at which mitosis could occur, note that cell types not present at start will have this as zero
            # unless the max statements are present to take info from pairs of types
            mitosisFactor = 2.0
            if i in [EPI_ME,PERI_ME]: mitosisFactor = 3.0    # more difficult to start mitosis for these cells
            if i in [MESENCH_N,MESENCH_O,MESENCH_EMT]:
                MITOSIS_VOLUME_LIST[i]=max(self.avgTypeVolume[MESENCH_N],self.avgTypeVolume[MESENCH_O],self.avgTypeVolume[MESENCH_EMT])*mitosisFactor
            elif i in [EPI_N,EPI_O,EPI_ME]:
                MITOSIS_VOLUME_LIST[i]=max(self.avgTypeVolume[EPI_N],self.avgTypeVolume[EPI_O],self.avgTypeVolume[EPI_ME])*mitosisFactor
            elif i in [PERI_N,PERI_O,PERI_ME]:
                MITOSIS_VOLUME_LIST[i]=max(self.avgTypeVolume[PERI_N],self.avgTypeVolume[PERI_O],self.avgTypeVolume[PERI_ME])*mitosisFactor
        print "\nCellType\tMitosisVolume"
        print "------------------------------------"
        for i in range(nCellTypes):
            print "%(type)9s\t%(mitVol)9.1f" % {'type':typeList[i], 'mitVol':MITOSIS_VOLUME_LIST[i]}
                #if i in [EPI_ME,PERI_ME]:
                    #MITOSIS_VOLUME_LIST[i]=(self.avgTypeVolume[i])*1.4
                #else: MITOSIS_VOLUME_LIST[i]=(self.avgTypeVolume[i])*2.0

        if self.useAvgVolumeForEachCellType:
            print "Assigning targetVolume of each cell to match cell-type averages."
            print "Assigning targetSurface of each cell based on targetVolume and desired shape.\n"
            for cell in self.cellList:
                if cell.type==WALL:                # frozen, so these only matter for display fields
                    cell.targetVolume=cell.volume
                    cell.lambdaVolume=0.0
                    cell.targetSurface=cell.surface
                    cell.lambdaSurface=0.0
                elif cell.type in [MESENCH_N, MESENCH_O, MESENCH_EMT]:
                    if self.useRandomVolumeForMesenchCells:
                        cell.targetVolume= random.uniform(0.05*MITOSIS_VOLUME_LIST[cell.type],0.95*MITOSIS_VOLUME_LIST[cell.type])*pfactor
                    else:
                        cell.targetVolume=self.avgTypeVolume[cell.type]*pfactor
                    cell.lambdaVolume=LV
                    if self.useInitTargetSurfaceBasedOnVolume or self.useRandomVolumeForMesenchCells:
                        cell.targetSurface=((cell.targetVolume*pfactor)**(1.0/2))*4.0 #cuboidal cells
                    else: cell.targetSurface=self.avgTypeSurface[cell.type]*pfactor
                    cell.lambdaSurface=LS
                elif cell.type in [EPI_N, EPI_O, EPI_ME]:
                    cell.targetVolume=self.avgTypeVolume[cell.type]*pfactor
                    cell.lambdaVolume=LV
                    if self.useInitTargetSurfaceBasedOnVolume:
                        cell.targetSurface=((self.avgTypeVolume[cell.type]*pfactor)**(1.0/2))*4.0 #cuboidal cells
                    else: cell.targetSurface=self.avgTypeSurface[cell.type]*pfactor
                    cell.lambdaSurface=LS
                elif cell.type in [PERI_N, PERI_O, PERI_ME]:
                    cell.targetVolume=self.avgTypeVolume[cell.type]*pfactor
                    cell.lambdaVolume=4*LV
                    if self.useInitTargetSurfaceBasedOnVolume:
                        cell.targetSurface=((self.avgTypeVolume[cell.type]*pfactor)**(1.0))*0.7 #flattened cells
                    else: cell.targetSurface=self.avgTypeSurface[cell.type]*pfactor
                    cell.lambdaSurface=LS
                    self.lengthConstraintPlugin.setLengthConstraintData(cell, 20*cell.lambdaSurface, cell.targetVolume*0.3)
                else:                     # Unknown type
                    cell.targetVolume=0
                    cell.lambdaVolume=0.0
                    cell.targetSurface=0
                    cell.lambdaSurface=0.0
                if verbose:
                    print "Cell "+str(cell.id)+" type = "+str(cell.type)+" targetVolume = " \
                            +str(cell.targetVolume)+" targetSurface = "+str(cell.targetSurface)
        else:
            print "Assigning targetVolume and targetSurface of each cell to match that cell's initial volume and surface area.\n"
            for cell in self.cellList:
                if cell.type==WALL:                # frozen, so these only matter for display fields
                    cell.targetVolume=cell.volume
                    cell.lambdaVolume=0.0
                    cell.targetSurface=cell.surface
                    cell.lambdaSurface=0.0
                elif cell.type in [MESENCH_N, MESENCH_O, MESENCH_EMT]:
                    if self.useRandomVolumeForMesenchCells:
                        cell.targetVolume= random.uniform(0.05*MITOSIS_VOLUME_LIST[cell.type],0.95*MITOSIS_VOLUME_LIST[cell.type])*pfactor
                        cell.targetSurface=((cell.targetVolume*pfactor)**(1.0/2))*4.0 #cuboidal cells
                    else:
                        cell.targetVolume=cell.volume*pfactor
                        cell.targetSurface=cell.surface*pfactor
                    cell.lambdaVolume=LV
                    cell.lambdaSurface=LS
                elif cell.type in [EPI_N, EPI_O, EPI_ME]:
                    cell.targetVolume=cell.volume*pfactor
                    cell.lambdaVolume=LV
                    cell.targetSurface=cell.surface*pfactor
                    cell.lambdaSurface=LS
                elif cell.type in [PERI_N, PERI_O, PERI_ME]:
                    cell.targetVolume=cell.volume*pfactor
                    cell.lambdaVolume=4*LV
                    cell.targetSurface=cell.surface*pfactor
                    cell.lambdaSurface=LS
                    self.lengthConstraintPlugin.setLengthConstraintData(cell, 20*cell.lambdaSurface, cell.targetVolume*0.3)
                else:                     # Unknown type
                    cell.targetVolume=0
                    cell.lambdaVolume=0.0
                    cell.targetSurface=0
                    cell.lambdaSurface=0.0
                if verbose:
                    print "Cell "+str(cell.id)+" type = "+str(cell.type)+" targetVolume = "\
                            +str(cell.targetVolume)+" targetSurface = "+str(cell.targetSurface)
    
        
    def step(self,mcs):      ####Defines how the cells change based on cell conditions and concentration gradients
        print "\n***** PalateModelCellBehaviors (step) ***************************"
        t0=time.clock()
        global SIMULATION_TIME
        timeBreakdownList = [0 for i in range(7)]
        timeBreakdownNames =["getInfo","apopBeh","apopDec","mitoDec","motiBeh","diffBeh","growBeh"]
        nSatisfyTimerFlag =[0 for x in range(nCellTypes)]
        nSatisfyVolumeFlag=[0 for x in range(nCellTypes)]
        nSatisfyAspectFlag=[0 for x in range(nCellTypes)]
        nClusterOfBaseType=[0 for x in range(nCellTypes)]
        nApoptosing = [0 for x in range(nCellTypes)]
        maxApoptosisProb = 0
        growthLists = [[] for x in range(nCellTypes)]
        matrixSecretionLists = [[] for x in range(nCellTypes)]
        motilityLists = [[] for x in range(nCellTypes)]
        secretionListsDict = {
             "SHH":[[] for x in range(nCellTypes)],
             "FGF10":[[] for x in range(nCellTypes)],
             "FGF7":[[] for x in range(nCellTypes)],
             "BMP2":[[] for x in range(nCellTypes)],
             "BMP4":[[] for x in range(nCellTypes)],
             "NOGGIN":[[] for x in range(nCellTypes)],
             "TGFb3":[[] for x in range(nCellTypes)],
             "EGF":[[] for x in range(nCellTypes)],
             "MMP":[[] for x in range(nCellTypes)]}
        apoptosisProbLists = [[] for x in range(nCellTypes)]
        apoptosisActionsList = []
        emtProbLists = [[] for x in range(nCellTypes)]
        eptProbLists = [[] for x in range(nCellTypes)]
        emtActionsList = []
        eptActionsList = []
        fppLambdaLists = [[] for x in range(nCellTypes)]
        fppTargetLists = [[] for x in range(nCellTypes)]
        fieldSHH=  CompuCell.getConcentrationField(self.simulator,"SHH")
        fieldFGF10=CompuCell.getConcentrationField(self.simulator,"FGF10")
        fieldFGF7=CompuCell.getConcentrationField(self.simulator,"FGF7")
        fieldBMP2=CompuCell.getConcentrationField(self.simulator,"BMP2")
        fieldBMP4=CompuCell.getConcentrationField(self.simulator,"BMP4")
        fieldNOGGIN=CompuCell.getConcentrationField(self.simulator,"NOGGIN")
        fieldTGFb3=CompuCell.getConcentrationField(self.simulator,"TGFb3")
        fieldEGF=CompuCell.getConcentrationField(self.simulator,"EGF")
        fieldMMP=CompuCell.getConcentrationField(self.simulator,"MMP")
        AC50TGFb3 = 70
        AC50EGF = 70
        # use functional form to get time-dependent fold change in EGFR levels
        self.ahrDrivenEGFRFoldChange = self.getAhrDrivenEGFRFoldChange(mcs)

        # ************* TEST ADDING A PLOT WINDOW ************************
        #self.pW.addDataPoint("D",mcs,mcs*mcs) # args are name, x, y
        #self.pW.showAllPlots()
        # ************* TEST ADDING A PLOT WINDOW ************************
        for compartmentList in self.clusterList:
            t0InsideLoop = time.clock()
	         # if needed, you can get number of compartments in a cluster using compartmentList.size()
	         # first collect information about the cluster (i.e., compartmentalized cell)
            clusterTimerFlag = True
            clusterApoptosisFlag=False
            clusterVolume = 0
            clusterTargetVolume = 0
            clusterBaseType = 0
            clusterId = 0
            clusterMaxAspect = 1
            clusterBaseCell = [cell for cell in CompartmentList(compartmentList)][0] #not exactly shorthand to get 1st cell in cluster
            for cell in CompartmentList(compartmentList):
                if cell.type in self.clusterBaseTypeList:
                    clusterBaseType = cell.type
                    clusterBaseCell = cell
                clusterVolume += cell.volume
                clusterTargetVolume += cell.targetVolume
                clusterId = cell.clusterId
                cellAttributes=CompuCell.getPyAttrib(cell)
                clusterTimerFlag = clusterTimerFlag and (cellAttributes['cell.timer.ectoDivision']<=0) #false if timer >0 for any compartment
                clusterApoptosisFlag = clusterApoptosisFlag or cellAttributes['cell.apoptosisFlag'] #true if apoptosisFlag is true for any
                if cellAttributes['cell.apoptosisFlag']: nApoptosing[cell.type]+=1  
                if (not clusterApoptosisFlag) and cell.type in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME,PERI_Na,PERI_Oa,PERI_MEa]: 
                    # slow calculation that is not needed for all cell types
                    if self.momentOfInertiaPlugin.getSemiaxes(cell)[0]>0:
                        aspect = self.momentOfInertiaPlugin.getSemiaxes(cell)[2]/self.momentOfInertiaPlugin.getSemiaxes(cell)[0]
                        if aspect > clusterMaxAspect: clusterMaxAspect = aspect
            timeBreakdownList[0]+=time.clock()-t0InsideLoop
            t0InsideLoop = time.clock()
            # ephrinB1SignalFlag=True if a MESENCH cell is in contact with another MESENCH cell
            # set this flag to always be zero to mimic ephrinB1 mutant
            ephrinB1geneOK = True
            neighborCellTypeList = self.getNeighborCellTypes(clusterBaseCell)
            ephrinB1SignalFlag = (ephrinB1geneOK and
                                  (clusterBaseCell.type in [MESENCH_N,MESENCH_O,MESENCH_EMT]) and 
                                  any(x in neighborCellTypeList for x in [MESENCH_N,MESENCH_O,MESENCH_EMT]))
            #if self.verbose and clusterBaseType==MESENCH_O: print "volume of Mesench_O cell",clusterId,"=",clusterVolume
            
	    if clusterApoptosisFlag:
		####### Apoptosis Behaviors BEGIN ######
                for cell in CompartmentList(compartmentList):
                    dV = (1.0*self.frequency/self.apoptosisTimeConstant)*cell.targetVolume
                    cell.targetVolume -= dV 
                    #print "APOPTOSIS: cell",cell.id,"of type",typeList[cell.type],"is apoptosing. targetVolume =",cell.targetVolume
                    actionString = "\tUndergoing apoptosis:\t cell %4d of type %11s; targetVolume= %.1f" % (cell.id, typeList[cell.type], cell.targetVolume)
                    apoptosisActionsList.append(actionString)
                    cell.targetSurface=4*sqrt(max(cell.targetVolume,0))
                    self.lengthConstraintPlugin.setLengthConstraintData(cell, 0, cell.targetVolume*0.3)  #gets rid of constraint if it existed
		####### Apoptosis Behaviors END ########
                timeBreakdownList[1]+=time.clock()-t0InsideLoop
                t0InsideLoop = time.clock()

	    else:
		####### Apoptosis Decisions BEGIN ######
                # choose which cells will apoptose (probabilistically - can have lots of adjustments to these probabilities)
                for cell in CompartmentList(compartmentList):
                    #if cell.type==PERI_N: print "neighborCellTypeList =",neighborCellTypeList
                    apoptosisProb = 0.0   # probability per mcs
                    pt = self.getCentroid(cell,_hexLattice = True)
                    if mcs > self.allowApoptosisTime and cell.type in [MESENCH_N,MESENCH_O,MESENCH_EMT,EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME,PERI_Na,PERI_Oa,PERI_MEa]:
                        # format of this list comprehension inside the 'any' statement will check to see if any members 
                        # of range(MESENCH,PERI_MEa+1) are found in the neighborCellTypeList. If not, then this is an isolated cell
                        # that has no contact with other cells or matrix
                        neighborCellTypeList = self.getNeighborCellTypes(cell)
                        if not any(x in neighborCellTypeList for x in range(MESENCH_N,PERI_MEa+1)): 
                            apoptosisProb +=0.1 # units are prob. per cell per mcs
                            #print "Increasing apoptosisProb for",typeList[cell.type],"cell: not in contact with any cell or matrix"  
                        elif self.TGFbetaFlag and cell.type in [PERI_Na,PERI_Oa,PERI_MEa]:
                            apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.0001, _AC50=AC50TGFb3)
                            apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.0001*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            # increases chances if not in contact with Medium
                            if not any(x in neighborCellTypeList for x in [MEDIUM]): 
                                #apoptosisProb +=0.01
                                apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.01, _AC50=AC50TGFb3)
                                apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.01*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            #    print "Increasing apoptosisProb for",typeList[cell.type],"cell: not in contact with MEDIUM"  
                            # and does so again if not in contact with Peri_N/O/ME
                            if not any(x in neighborCellTypeList for x in [PERI_N,PERI_O,PERI_ME]): 
                                #apoptosisProb +=0.01
                                apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.01, _AC50=AC50TGFb3)
                                apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.01*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            #    print "Increasing apoptosisProb for",typeList[cell.type],"cell: not in contact with [PERI_N,PERI_ME]"  
                        elif self.TGFbetaFlag and cell.type in [PERI_N,PERI_O,PERI_ME]:
                            apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.0001, _AC50=AC50TGFb3)
                            apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.0001*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            # if not in contact with Medium or Peri_N/2a
                            if not any(x in neighborCellTypeList for x in [MEDIUM,PERI_Na,PERI_Oa,PERI_MEa]): 
                                #apoptosisProb +=0.01
                                apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.01, _AC50=AC50TGFb3)
                                apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.01*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            #    print "Increasing apoptosisProb for",typeList[cell.type],"cell: not in contact with [MEDIUM,PERI_Na,PERI_MEa]"  
                            #elif self.TGFbetaFlag and cell.type in [PERI_N,PERI_ME]:  #FIX - I don't think program will ever get to this statement
                            # if not in contact with some other part of a PERI cell
                            elif not any(x in neighborCellTypeList for x in [PERI_Na,PERI_Oa,PERI_MEa,PERI_N,PERI_O,PERI_ME]): 
                                #apoptosisProb +=0.01
                                apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.01, _AC50=AC50TGFb3)
                                apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.01*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            #    print "Increasing apoptosisProb for",typeList[cell.type],"cell: not in contact with any other PERI cell part"  
                        elif self.TGFbetaFlag and cell.type in [EPI_N,EPI_O,EPI_ME]:
                            apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.00002, _AC50=AC50TGFb3)
                            apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.00002*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            # if not in contact with Medium or any part of a PERI cluster
                            if not any(x in neighborCellTypeList for x in [MEDIUM,PERI_Na,PERI_Oa,PERI_MEa,PERI_N,PERI_O,PERI_ME,TJ,TJm]): 
                                apoptosisProb +=0.000
                                apoptosisProb +=self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.002, _AC50=AC50TGFb3)
                                apoptosisProb -=self.hillFunction(fieldEGF.get(pt),     _maxRate=0.002*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                            #    print "Increasing apoptosisProb for",typeList[cell.type],"cell: not in contact with MEDIUM or any PERI cell part"  
                        #note that the following will set apoptosisFlag=True, but will never reverse this decision
                        #option to increase all apoptosis probabilities
                        apoptosisProb *= 5
                        # scale appropriately so rate of cell apoptosis is independent of frequency with which steppable is called
                        apoptosisProb = 1 - ((1-apoptosisProb)**(self.frequency)) # note that '**' is raised to a power in python
                        (apoptosisProbLists[cell.type]).append(apoptosisProb)
                        if random.random()<apoptosisProb: 
                            (CompuCell.getPyAttrib(cell))['cell.apoptosisFlag']=True
                            actionString = "\tCommitted to apoptosis:\t cell %4d of type %11s" % (cell.id, typeList[cell.type])
                            apoptosisActionsList.append(actionString)
                        if apoptosisProb>maxApoptosisProb: maxApoptosisProb=apoptosisProb
		####### Apoptosis Decisions END ########
                timeBreakdownList[2]+=time.clock()-t0InsideLoop
                t0InsideLoop = time.clock()

                ####### Mitosis Decisions BEGIN ######
                clusterVolumeFlag =  (clusterVolume > MITOSIS_VOLUME_LIST[clusterBaseType])
                clusterAspectFlag =  ((not self.useAspectRatioForMitosis)                   or
                                     (clusterBaseType in [MESENCH_N,MESENCH_O,MESENCH_EMT]) or
                                     (clusterBaseType==EPI_N  and clusterMaxAspect>3.0)     or
                                     (clusterBaseType==EPI_O  and clusterMaxAspect>3.0)     or
                                     (clusterBaseType==EPI_ME  and clusterMaxAspect>2.0)    or
                                     (clusterBaseType==PERI_N and clusterMaxAspect>5.0)     or
                                     (clusterBaseType==PERI_O and clusterMaxAspect>5.0)     or
                                     (clusterBaseType==PERI_ME and clusterMaxAspect>3.0))
		clusterTypeFlag = (not (clusterBaseType in self.doNotDivideTypeList))

	        if clusterTimerFlag:  nSatisfyTimerFlag[clusterBaseType]+=1
                if clusterVolumeFlag: nSatisfyVolumeFlag[clusterBaseType]+=1
                if clusterAspectFlag: nSatisfyAspectFlag[clusterBaseType]+=1
	        nClusterOfBaseType[clusterBaseType]+=1

                mitosisFlag = clusterTypeFlag and clusterTimerFlag and clusterVolumeFlag and clusterAspectFlag
                for cell in CompartmentList(compartmentList):
	            CompuCell.getPyAttrib(cell).update({'cell.mitosisFlag':mitosisFlag})
	        if self.verbose and mitosisFlag:
                    print "ClusterId=",clusterId," clusterBaseType=",typeList[clusterBaseType],\
			"clusterVolume=",clusterVolume,"maxAspect=",clusterMaxAspect
                    print "\tFLAGS (volume, timer, aspect)=", clusterVolumeFlag, clusterTimerFlag, clusterAspectFlag
                ####### Mitosis Decisions END   ######
                timeBreakdownList[3]+=time.clock()-t0InsideLoop
                t0InsideLoop = time.clock()

                ####### Motility Behaviors BEGIN######
                for cell in CompartmentList(compartmentList):
                    pt = self.getCentroid(cell,_hexLattice = True)
                    if self.reduceMotilityOfPeriCells and mcs>self.mcsAtWhichToDecreasePeriCellMotility: 
                        if cell.type in [PERI_N,PERI_O,PERI_ME,PERI_Na,PERI_Oa,PERI_MEa]:
                            #cell.fluctAmpl=1    #make these cell types nearly immotile
                            cell.fluctAmpl = max(0,2 + (self.hillFunction(fieldTGFb3.get(pt),   _maxRate=8, _AC50=AC50TGFb3) -
                                                  self.hillFunction(fieldEGF.get(pt),     _maxRate=8*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)))
                            (motilityLists[cell.type]).append(cell.fluctAmpl)
                    #if self.TGFbetaFlag and mcs>self.mcsAtWhichToIncreasePeriCellMotility and cell.type in [PERI_ME,PERI_MEa]:
                        #cell.fluctAmpl=16   #make these cell types hypermotile
                ####### Motility Behaviors END######
                timeBreakdownList[4]+=time.clock()-t0InsideLoop
                t0InsideLoop = time.clock()

                ####### Differentiation Behaviors BEGIN######
                for cell in CompartmentList(compartmentList):
                    pt = self.getCentroid(cell,_hexLattice = True)
                    epiSwitchRate = 0.001
                    if mcs > self.allowEmtTime and cell.type in [EPI_N,EPI_O,EPI_ME]:
                        emtProb=0.0   #epithelial-to-mesenchymal transition probability
                        eptProb=0.0   #epithelial-to-periderm transition probability
                        # if not in contact with Medium or any part of a PERI cluster, but in contact with MESENCH or MATRIX
                        if not self.isBoundaryCell(cell,[MEDIUM,PERI_Na,PERI_Oa,PERI_MEa,PERI_N,PERI_O,PERI_ME,TJ,TJm]):
                            if self.isBoundaryCell(cell,[MESENCH_N,MESENCH_O,MESENCH_EMT,MATRIX]): 
                                emtProb += 0.00001
                                emtProb += self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.001, _AC50=AC50TGFb3)
                                emtProb -= self.hillFunction(fieldEGF.get(pt),     _maxRate=0.001*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF)
                        # if not in contact with MESENCH or MATRIX, but in contact with Medium or any part of a PERI cluster
                        else:
                            if not self.isBoundaryCell(cell,[MESENCH_N,MESENCH_O,MESENCH_EMT,MATRIX]):
                                eptProb += 0.0001
                                if self.isBoundaryCell(cell,[MEDIUM]): eptProb +=0.01      #higher prob if in direct contact with Medium                          
                        # scale appropriately so rates of differentiation is independent of frequency with which steppable is called
                        emtProb = 1 - ((1-emtProb)**(self.frequency)) # note that '**' is raised to a power in python
                        eptProb = 1 - ((1-eptProb)**(self.frequency)) # note that '**' is raised to a power in python
                        (emtProbLists[cell.type]).append(emtProb)
                        (eptProbLists[cell.type]).append(eptProb)
                        if random.random()<emtProb:
                            print "EMT:       cell",cell.id,"from type",typeList[cell.type],"to",typeList[MESENCH_EMT]
                            actionString = "\tEMT of cell "+str(cell.id)+" from type "+typeList[cell.type]+" to type "+typeList[MESENCH_EMT]
                            emtActionsList.append(actionString)
                            #print "--------------- targetVolume=",cell.targetVolume,"lambdaVolume=",cell.lambdaVolume
                            cell.type=MESENCH_EMT
                        if random.random()<eptProb:
                            if   cell.type==EPI_N:  newCellType=PERI_N
                            elif cell.type==EPI_O:  newCellType=PERI_O
                            elif cell.type==EPI_ME: newCellType=PERI_ME
                            print "EPT:       cell",cell.id,"from type",typeList[cell.type],"to",typeList[newCellType]
                            actionString = "\tEPT of cell "+str(cell.id)+" from type "+typeList[cell.type]+" to type "+typeList[newCellType]
                            eptActionsList.append(actionString)
                            #print "--------------- targetVolume=",cell.targetVolume,"lambdaVolume=",cell.lambdaVolume
                            cell.type=newCellType
                        #elif cell.type==EPI_N:
                            # differentiate to type EPI_ME with higher prob if [FGF10] is high
                        #    transProb = self.hillFunction(fieldFGF10.get(pt),   _maxRate=epiSwitchRate, _AC50=20, _n=10)
                        #    transProb = 1 - ((1-transProb)**(self.frequency)) # note that '**' is raised to a power in python
                        #    if random.random()<transProb: cell.type=EPI_ME  
                        #elif cell.type==EPI_ME:
                            # differentiate to type EPI_N with higher prob if [FGF10] is low
                        #    transProb = 0.1*epiSwitchRate*(1 - self.hillFunction(fieldFGF10.get(pt),   _maxRate=1, _AC50=20, _n=10))
                        #    transProb = 1 - ((1-transProb)**(self.frequency)) # note that '**' is raised to a power in python
                        #    if random.random()<transProb: cell.type=EPI_N  
                                      
                ####### Differentiation Behaviors END######
                timeBreakdownList[5]+=time.clock()-t0InsideLoop
                t0InsideLoop = time.clock()

                ####### Growth and Matrix Secretion Behaviors BEGIN######
                if mcs <= self.timeToStopGrowth:
                    if clusterBaseCell.type in [MESENCH_N, MESENCH_O, MESENCH_EMT]:
                        cell=clusterBaseCell
                        pt = self.getCentroid(cell,_hexLattice = True)
                        neighborCellTypeList = self.getNeighborCellTypes(cell)
                        # ephrinSignalFlag=True if this MESENCH cell is in contact with another MESENCH cell
                        # set this flag to always be zero to mimic ephrinB1 mutant
                        #ephrinB1SignalFlag = any(x in neighborCellTypeList for x in [MESENCH_N,MESENCH_EMT])
                        growthProb = 0.5
                        if not ephrinB1SignalFlag: growthProb = 0.1
                        growthRate = 0.0
                        mesenchMaxGrowthRate = 0.4
                        if self.useBM: mesenchMaxGrowthRate = 3.0*mesenchMaxGrowthRate 
                        if fieldBMP2.get(pt)+fieldBMP4.get(pt) > 0:
                            fBMP = (fieldBMP2.get(pt)+fieldBMP4.get(pt)-fieldNOGGIN.get(pt))/(fieldBMP2.get(pt)+fieldBMP4.get(pt))
                            if fBMP>0:
                                growthRate += self.hillFunction(fieldBMP2.get(pt)*fBMP, _maxRate=mesenchMaxGrowthRate, _AC50=20)
                        if cell.targetVolume<(MITOSIS_VOLUME_LIST[cell.type])*1.5:
                            deltaTargetVolume = growthRate*binomial(self.frequency, growthProb) # replaces random.randint(0,1) to apply to all self.frequency
                            growthLists[cell.type].append(deltaTargetVolume)
                            cell.targetVolume += deltaTargetVolume
                            cell.targetSurface=4*sqrt(max(cell.targetVolume,0))
                        if self.useMatrix:
                            matrixSecretionProb = 0.5
                            if not ephrinB1SignalFlag: matrixSecretionProb = 0.25
                            matrixSecretionRate = self.hillFunction(fieldBMP2.get(pt),   _maxRate=mesenchMaxGrowthRate, _AC50=20)
                            # amount of matrix to secrete (or occasionally ingest); cannot be more than cell's targetVolume
                            volumeToTransfer = matrixSecretionRate*(-0.1*self.frequency 
                                                                    + 0.4*binomial(self.frequency, matrixSecretionProb))
                            volumeToTransfer = min(volumeToTransfer, cell.targetVolume)
                            # check if there are any adjacent cells of type MATRIX
                            matrixCell=self.getBoundaryCell(cell,[MATRIX])
                            if matrixCell:
                                matrixCell.targetVolume += volumeToTransfer
                                # matrix "cells" are nucleated with very high lambdaVolume and then reduced later here
                                #if matrixCell.targetVolume > 10: matrixCell.lambdaVolume=cell.lambdaVolume
                                matrixSecretionLists[cell.type].append(volumeToTransfer)
                                cell.targetVolume -= volumeToTransfer
                                cell.targetSurface=4*sqrt(max(cell.targetVolume,0))
                            elif random.random()<volumeToTransfer:   # FIX this probability is used to control growth rate
                                matrixCell=self.nucleateMatrixCell(cell)  # nucleates a new MATRIX cell and sets targetVolume=1
                                if matrixCell and cell.targetVolume>1:
                                    matrixSecretionLists[cell.type].append(1)
                                    cell.targetVolume -= 1
                                    cell.targetSurface=4*sqrt(max(cell.targetVolume,0))
                    elif clusterBaseCell.type in [EPI_N, EPI_O, EPI_ME]:
                        cell=clusterBaseCell
                        pt = self.getCentroid(cell,_hexLattice = True)
                        #growthRateMultiplier = 1.0 
                        #if self.useBM: growthRateMultiplier = 0.9 # explicit BM slows down MESENCH growth, so slow EPI/PERI too 
                        growthRate = 0.016 # was 0.01
                        growthRate += self.hillFunction(fieldFGF10.get(pt),   _maxRate=0.20, _AC50=10)
                        growthRate += self.hillFunction(fieldFGF7.get(pt),    _maxRate=0.20, _AC50=150)
                        growthRate -= self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.08, _AC50=AC50TGFb3, _n=4)
                        growthRate += self.hillFunction(fieldEGF.get(pt),     _maxRate=0.08*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF,   _n=4)
                        growthRate = max(growthRate,0) # make sure it is not less than zero
                        if cell.targetVolume<(MITOSIS_VOLUME_LIST[cell.type])*2.5:
                            deltaTargetVolume = growthRate*binomial(self.frequency,0.5)
                            growthLists[cell.type].append(deltaTargetVolume)
                            cell.targetVolume += deltaTargetVolume
                            cell.targetSurface=4*sqrt(max(cell.targetVolume,0))  #want cuboidal EPI cells
                        

                    elif clusterBaseCell.type in [PERI_N, PERI_O, PERI_ME]: # handles compartments of PERI cells too
                        pt = self.getCentroid(cell,_hexLattice = True)
                        #growthRateMultiplier = 1.0 
                        #if self.useBM: growthRateMultiplier = 0.7 # explicit BM slows down MESENCH growth, so slow EPI/PERI too 
                        growthRate = 0.01 
                        growthRate += self.hillFunction(fieldFGF10.get(pt),   _maxRate=0.12, _AC50=10)
                        growthRate += self.hillFunction(fieldFGF7.get(pt),    _maxRate=0.12, _AC50=150)
                        growthRate -= self.hillFunction(fieldTGFb3.get(pt),   _maxRate=0.08, _AC50=AC50TGFb3, _n=4)
                        growthRate += self.hillFunction(fieldEGF.get(pt),     _maxRate=0.08*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF,   _n=4)
                        growthRate = max(growthRate,0) # make sure it is not less than zero
                        if clusterTargetVolume<(MITOSIS_VOLUME_LIST[clusterBaseCell.type])*2.5:
                            deltaTargetVolume = growthRate*binomial(self.frequency,0.5)
                            growthLists[clusterBaseCell.type].append(deltaTargetVolume)
                            clusterBaseCell.targetVolume += deltaTargetVolume
                            # see if cluster has an apical domain; growth differs if it does
                            apicalCell = 0
                            #for i in range(compartmentList.size()):
                            for subcell in CompartmentList(compartmentList):
                                if subcell.type in [PERI_Na,PERI_Oa,PERI_MEa]: apicalCell = subcell
                            if apicalCell:   # settings for cells that do have apical polarization
                                clusterBaseCell.targetSurface=0.9*clusterBaseCell.targetVolume   #want flattened PERI cells
                                self.lengthConstraintPlugin.setLengthConstraintData(clusterBaseCell, 
                                                                                    20*clusterBaseCell.lambdaSurface, 
                                                                                    clusterBaseCell.targetVolume*0.5)
                                growthLists[apicalCell.type].append(deltaTargetVolume)
                                apicalCell.targetVolume = clusterBaseCell.targetVolume
                                apicalCell.targetSurface = apicalCell.targetVolume
                                self.lengthConstraintPlugin.setLengthConstraintData(apicalCell, 
                                                                                    20*apicalCell.lambdaSurface, 
                                                                                    apicalCell.targetVolume*0.5)
                            else:           # settings for cells that are not apically polarized
                                clusterBaseCell.targetSurface=0.7*clusterBaseCell.targetVolume   #want flattened PERI cells
                                self.lengthConstraintPlugin.setLengthConstraintData(clusterBaseCell, 
                                                                                    20*clusterBaseCell.lambdaSurface, 
                                                                                    clusterBaseCell.targetVolume*0.3)
                else: growthRate =0
                ####### Basement Membrane Behaviors BEGIN######
                # secrete new BM if an EPI cell is in contact with a MESENCH cell or with MATRIX
                if self.useBM:
                    if (clusterBaseCell.type in [EPI_N, EPI_O, EPI_ME]):
                       # Secrete/nucleate a BM cell whenever and EPI cell is in contact with MESENCH or MATRIX
                       bmSecretionProb = 1.0
                       if not self.isBoundaryCell(cell,[MESENCH_N,MESENCH_O,MESENCH_EMT,MATRIX]): 
                           bmSecretionProb = 0.0
                       # to maintain same prob per mcs regardless of frequency with which steppable is called
                       # change Prob as described below (not really necessary when Prob = 1.0 
                       bmSecretionProb = 1 - ((1-bmSecretionProb)**(self.frequency)) # note that '**' is raised to a power in python
                       if random.random()<bmSecretionProb: bmCell=self.nucleateBM(cell)  # nucleates a new BM cell and sets targetVolume
                    if (clusterBaseCell.type in [BM]):
                       # eliminate BM "cells" that have only one compartment
                       if (compartmentList.size()<2):
                          clusterBaseCell.targetVolume = 0
                       # represent MMP cleavage of BM as probabilistic elimination of BM cells
                       kcat = 0.005
                       bmCleavageProb = kcat*fieldMMP.get(pt)
                       if random.random()<bmCleavageProb: clusterBaseCell.targetVolume = 0
                ####### Basement Membrane Behaviors END######

                ####### Growth and Matrix Secretion Behaviors END######
                timeBreakdownList[6]+=time.clock()-t0InsideLoop
                t0InsideLoop = time.clock()
                
                ####### Secretion Behaviors BEGIN####
                for cell in CompartmentList(compartmentList):
                    if cell.type in [MESENCH_N,MESENCH_O,MESENCH_EMT,EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME,PERI_Na,PERI_Oa,PERI_MEa]:
                        pt = self.getCentroid(cell,_hexLattice = True)
                        if cell.type in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME,PERI_Na,PERI_Oa,PERI_MEa]:
                            if 100>=mcs>0:
                                # set initial conditions; SHH initially secreted by EPI_ME cells
                                if self.geneStatusDict['SHH'] and cell.type in [EPI_ME,PERI_ME,PERI_MEa]:
                                    self.getFieldSecretor("SHH").secreteInsideCell(cell,10)  
                                    ((secretionListsDict['SHH'])[cell.type]).append(0)
                                # set initial conditions; NOGGIN secreted by Epi_O cells
                                if self.geneStatusDict['NOGGIN'] and cell.type in [EPI_O,PERI_O,PERI_Oa]:
                                    self.getFieldSecretor("NOGGIN").secreteInsideCell(cell,10)
                                    ((secretionListsDict['NOGGIN'])[cell.type]).append(10)
                                # set initial conditions; TGFb3 secreted by Epi_ME cells
                                if self.geneStatusDict['TGFb3'] and cell.type in [EPI_ME,PERI_ME,PERI_MEa]:
                                    self.getFieldSecretor("TGFb3").secreteInsideCell(cell,10)
                                    ((secretionListsDict['TGFb3'])[cell.type]).append(10)
                                # set initial conditions; EGF secreted by Epi_ME cells
                                if self.geneStatusDict['EGF'] and cell.type in [EPI_ME,PERI_ME,PERI_MEa]:
                                    self.getFieldSecretor("EGF").secreteInsideCell(cell,0)
                                    ((secretionListsDict['EGF'])[cell.type]).append(0)
                            elif mcs>100:
                                # EPI_ME (those in MEE) cells secrete SHH in response to FGF10 and BMP4
                                s = 1.0   # used to scale SHH secretion to find right level for current model
                                secreteRate  =self.hillFunction(fieldFGF10.get(pt),   _maxRate=100*s,  _AC50=20)  #activation
                                if fieldBMP2.get(pt)+fieldBMP4.get(pt) > 0:
                                    fBMP = (fieldBMP2.get(pt)+fieldBMP4.get(pt)-fieldNOGGIN.get(pt))/(fieldBMP2.get(pt)+fieldBMP4.get(pt))
                                    if fBMP>0:
                                        secreteRate +=self.hillFunction(fieldBMP4.get(pt)*fBMP,    _maxRate=50*s,   _AC50=20)  #activation
                                secreteRate -=self.hillFunction(fieldFGF7.get(pt),    _maxRate=100*s,  _AC50=10)  #inhibition
                                secreteRate = max(secreteRate,0) # don't let it become negative
                                if self.geneStatusDict['SHH']:
                                    self.getFieldSecretor("SHH").secreteInsideCell(cell,secreteRate)   
                                    ((secretionListsDict['SHH'])[cell.type]).append(secreteRate)
                                # Should all EPI cells secrete TGFb3 in response to difference in BMP4 and FGF10 signals???
                                #secreteRate  =self.hillFunction(fieldBMP4.get(pt),   _maxRate=50, _AC50=20) # uses default _n=1
                                #secreteRate -=self.hillFunction(fieldFGF10.get(pt),    _maxRate=50,  _AC50=20)
                                #if secreteRate>0: self.getFieldSecretor("TGFb3").secreteInsideCell(cell,secreteRate)
                                # constant, but slow secretion of TGFb3 and EGF by cells in MEE; the two signals
                                # are mutually inhibitory, which should produce switch like behavior
                                # random #'s are included so system could fall to either side of the 'switch'
                                defaultRate = 15
                                # OLD METHOD FOR PUTTING IN EFFECTS of AHR ACTIVATION
                                #ahrDrivenRate = self.ahrActivity*1.0
                                #secreteRateTGFb3 = max(0,(defaultRate + random.random())*s - self.hillFunction(fieldEGF.get(pt),   
                                #   _maxRate=(defaultRate-3)*s*self.ahrDrivenEGFRFoldChange, _AC50=AC50EGF, _n=4))  # mutual inhibition
                                #secreteRateEGF   = (defaultRate + ahrDrivenRate + random.random())*s - self.hillFunction(fieldTGFb3.get(pt), 
                                #   _maxRate=(defaultRate-3)*s, _AC50=AC50TGFb3, _n=4)  # mutual inhibition
                                # NEW METHOD FOR PUTTING IN EFFECTS of AHR ACTIVATION
                                KdEGF = 50
                                KdTGFb3 = 50
                                nSwitch = 4
                                #receptorEGF   = (4**(1/nSwitch))*AC50TGFb3*(1.3)*(self.ahrDrivenEGFRFoldChange)
                                receptorEGF = self.EGFR_0*(self.ahrDrivenEGFRFoldChange)
                                #receptorTGFb3 = (4**(1/nSwitch))*AC50EGF*(1.3)*(1.2)
                                receptorTGFb3 = self.TGFBR_0
                                boundReceptorEGF   =   receptorEGF*fieldEGF.get(pt)/(KdEGF + fieldEGF.get(pt))
                                boundReceptorTGFb3 = receptorTGFb3*fieldTGFb3.get(pt)/(KdTGFb3 + fieldTGFb3.get(pt))
                                   # OLD METHOD THAT USED SAME AC50's for bound receptor as for total ligand
                                   #secreteRateTGFb3 = (defaultRate + random.random())*s - self.hillFunction(boundReceptorEGF,
                                   #_maxRate=defaultRate*s, _AC50=AC50EGF, _n=4)  # mutual inhibition
                                   #secreteRateEGF = (defaultRate + random.random())*s - self.hillFunction(boundReceptorTGFb3,
                                   #_maxRate=defaultRate*s, _AC50=AC50TGFb3, _n=4)  # mutual inhibition
                                secreteRateTGFb3 = (defaultRate + random.random())*s - self.hillFunction(boundReceptorEGF,
                                    _maxRate=defaultRate*s, _AC50=70, _n=4)  # mutual inhibition
                                secreteRateEGF = (defaultRate + random.random())*s - self.hillFunction(boundReceptorTGFb3,
                                    _maxRate=defaultRate*s, _AC50=70, _n=4)  # mutual inhibition
                                # EPI cells secrete a basal level of MMP + an amount that depends on TGFb3
                                secreteRateMMP   = 1.0 + self.hillFunction(fieldTGFb3.get(pt), _maxRate=3.0, _AC50=AC50TGFb3, _n=1)
                                # test effect of a pulse of expression. Can it flip the switch?
                                #if 800 > mcs > 700: secreteRateEGF = 3*defaultRate
                                if self.geneStatusDict['TGFb3'] and cell.type in [EPI_ME,PERI_ME,PERI_MEa]:
                                    self.getFieldSecretor("TGFb3").secreteInsideCell(cell,secreteRateTGFb3)
                                    ((secretionListsDict['TGFb3'])[cell.type]).append(secreteRateTGFb3)  
                                if self.geneStatusDict['EGF'] and cell.type in [EPI_ME,PERI_ME,PERI_MEa]:
                                    self.getFieldSecretor("EGF").secreteInsideCell(cell,secreteRateEGF)
                                    ((secretionListsDict['EGF'])[cell.type]).append(secreteRateEGF)  
                                # don't let PERI cells secrete MMP
                                if self.geneStatusDict['MMP'] and cell.type in [EPI_ME,EPI_O,EPI_N]:
                                    self.getFieldSecretor("MMP").secreteInsideCell(cell,secreteRateMMP)
                                    ((secretionListsDict['MMP'])[cell.type]).append(secreteRateMMP)  
                                # NOGGIN continues to be secreted by Epi_O cells; no literature guidance on feedbacks
                                if self.geneStatusDict['NOGGIN'] and cell.type in [EPI_O,PERI_O,PERI_Oa]:
                                    self.getFieldSecretor("NOGGIN").secreteInsideCell(cell,10)
                                    ((secretionListsDict['NOGGIN'])[cell.type]).append(10)   
                        elif cell.type in [MESENCH_N,MESENCH_O,MESENCH_EMT]:
                            if 100>=mcs>0:
                                # set initial conditions; BMP4 secretion is key to starting cascasde
                                if self.geneStatusDict['BMP4']:
                                    self.getFieldSecretor("BMP4").secreteInsideCell(cell,30)  # just a way to get this started
                                    ((secretionListsDict['BMP4'])[cell.type]).append(30)
                            elif mcs>100:
                                # MESENCH cells secrete FGF10 in response to SHH; cross-talk with ephrinB1 signals
                                maxSecreteRate = 30
                                if not ephrinB1SignalFlag: maxSecreteRate = 4
                                secreteRate=self.hillFunction(fieldSHH.get(pt), _maxRate=maxSecreteRate, _AC50=20, _n=1)
                                if self.geneStatusDict['FGF10']:
                                    self.getFieldSecretor("FGF10").secreteInsideCell(cell,secreteRate)
                                    ((secretionListsDict['FGF10'])[cell.type]).append(secreteRate)    
                                # MESENCH cells secrete BMP2 in response to SHH; cross-talk with ephrinB1 signals
                                secreteRate=self.hillFunction(fieldSHH.get(pt), _maxRate=maxSecreteRate, _AC50=20, _n=1)
                                if self.geneStatusDict['BMP2']:
                                    self.getFieldSecretor("BMP2").secreteInsideCell(cell,secreteRate)    
                                    ((secretionListsDict['BMP2'])[cell.type]).append(secreteRate)
                                # MESENCH cells secrete BMP4 with a self-induction loop but at a DECREASING rate in response to SHH; 
                                # cross-talk with ephrin B1 signals???
                                if fieldBMP2.get(pt)+fieldBMP4.get(pt) > 0:
                                    fBMP = (fieldBMP2.get(pt)+fieldBMP4.get(pt)-fieldNOGGIN.get(pt))/(fieldBMP2.get(pt)+fieldBMP4.get(pt))
                                    if fBMP>0:
                                        secreteRate =self.hillFunction(fieldBMP4.get(pt)*fBMP,    _maxRate=1.5*maxSecreteRate,   _AC50=10)  #activation
                                    else:
                                        secreteRate =0
                                secreteRate -=self.hillFunction(fieldSHH.get(pt),     _maxRate=maxSecreteRate,     _AC50=20, _n=1) #inhibition by Shh
                                secreteRate = max(secreteRate,0)
                                # MESENCH cells secrete BMP4 in response to SHH; cross-talk with ephrinB1 signals
                                #secreteRate= self.hillFunction(fieldSHH.get(pt), _maxRate=maxSecreteRate, _AC50=20, _n=1) #inhibition by Shh
                                if self.geneStatusDict['BMP4']:
                                    self.getFieldSecretor("BMP4").secreteInsideCell(cell,secreteRate)
                                    ((secretionListsDict['BMP4'])[cell.type]).append(secreteRate)
                                # MESENCH cells secrete a basal level of MMP
                                secreteRate = 1.0
                                if self.useBM and self.geneStatusDict['MMP']:
                                   self.getFieldSecretor("MMP").secreteInsideCell(cell,secreteRate)
                                   ((secretionListsDict['MMP'])[cell.type]).append(secreteRate)
                            if cell.type in [MESENCH_N]:
                                if 100>=mcs>0:
                                    # set initial conditions
                                    if self.geneStatusDict['FGF7']:
                                        self.getFieldSecretor("FGF7").secreteInsideCell(cell,20)  
                                        ((secretionListsDict['FGF7'])[cell.type]).append(0)
                                elif mcs>100:
                                    secreteRate  =self.hillFunction(fieldFGF7.get(pt), _maxRate=20,  _AC50=10, _n=1)  #auto-induction
                                    secreteRate -=self.hillFunction(fieldSHH.get(pt),  _maxRate=20,  _AC50=20, _n=1)  #inhibition
                                    secreteRate = max(secreteRate,0)
                                    if self.geneStatusDict['FGF7']:
                                        self.getFieldSecretor("FGF7").secreteInsideCell(cell,secreteRate)
                                        ((secretionListsDict['FGF7'])[cell.type]).append(secreteRate)
                        
                                
                ####### Secretion Behaviors END######
                
                ####### Matrix Elasticity Behaviors BEGIN####
                if self.useMatrixElasticity and clusterBaseCell.type in [MESENCH_N,MESENCH_O,MESENCH_EMT]:
                    cell = clusterBaseCell
                    #fppdListSize = sum(1 for _ in FocalPointPlasticityDataList(self.focalPointPlasticityPlugin, cell))
                    for fppd in FocalPointPlasticityDataList(self.focalPointPlasticityPlugin, cell):
                        d = self.getFppLinkDistance(cell,fppd)
                        newTargetDistance = fppd.targetDistance
                        newLambdaDistance = fppd.lambdaDistance + (self.fppLambdaMax - fppd.lambdaDistance)/self.fppTimeConstant
                        if fppd.lambdaDistance==0:
                            newTargetDistance = self.getFppLinkDistance(cell,fppd)
                        else:
                            newTargetDistance = fppd.targetDistance
                        #lambda = fppd 
                        self.focalPointPlasticityPlugin.setFocalPointPlasticityParameters\
                            (cell, fppd.neighborAddress, newLambdaDistance, newTargetDistance, 2.0*newTargetDistance)
                        (fppTargetLists[cell.type]).append(fppd.targetDistance)
                        (fppLambdaLists[cell.type]).append(fppd.lambdaDistance)
                ####### Matrix Elasticity Behaviors END######
                
        #******* Print out some info here on the cell behaviors ********
        print "\nGrowth Rates, i.e. DeltaTargetVolumes"
        self.printSummary(growthLists,"%.2f")
            
        print "\nMatrix Secretion/Degradation Rates, i.e. DeltaTargetVolumes"
	self.printSummary(matrixSecretionLists,"%.2f")

        print "\nSignal Secretion Rates"
        for k in secretionListsDict.keys():
            print "  ",k,"- - - - - - - - - - - -"
            self.printSummary(secretionListsDict[k],"%3.0f")
        
        print "\nModified Fluctuation Amplitudes"
        self.printSummary(motilityLists,"%.1f")

        print "\nApoptosis Probabilities (per cell per",self.frequency,"mcs)"
        self.printSummary(apoptosisProbLists,"%.4f")
        print "\nApoptosis Decisions/Behaviors"
        for entry in apoptosisActionsList: print entry
        
        print "\nEMT Probabilities (per cell per",self.frequency,"mcs)"
        self.printSummary(emtProbLists,"%.4f")
        print "\nEMT Decisions/Behaviors"
        for entry in emtActionsList: print entry

        print "\nEPT Probabilities (per cell per",self.frequency,"mcs)"
        self.printSummary(eptProbLists,"%.4f")
        print "\nEPT Decisions/Behaviors"
        for entry in eptActionsList: print entry

        print "\nFPP Link Parameters"
        print "lambaDistances:"
        self.printSummary(fppLambdaLists,"%.1f")
        print "targetDistances:"
        self.printSummary(fppTargetLists,"%.1f")

        if False:
	    for i in range(nCellTypes):
                nSatisfyFlags = [nSatisfyTimerFlag[i], nSatisfyVolumeFlag[i], nSatisfyAspectFlag[i]]
	        print "Cell type ",typeList[i],"N=",nClusterOfBaseType[i]," (N_timer, N_volume, N_aspect): ",nSatisfyFlags

	#following command will print out a formatted list of all the avg cell volumes and surface areas
	if mcs%(100*self.frequency)==0: self.calculateAvgPropertiesByType()
        if self.verbose: 
            print "MAXIMUM PROBABILITY OF ANY CELL TO START APOPTOSIS =",maxApoptosisProb
            print "self.TGFbetaFlag =",self.TGFbetaFlag
	    print "Number of cells apoptosing:"
	    for i in range(nCellTypes):
                print typeList[i],":",nApoptosing[i]
        print "\n\t\t\tTime taken for individual parts of loop over clusters:"
	timeSum = 0
	for i in range(7):
            print "\t\t\t\t",timeBreakdownNames[i],"=",timeBreakdownList[i]
            timeSum += timeBreakdownList[i]
        print "\t\t\t\t    SUM =",timeSum
        print "\t\t\tTime Taken=", time.clock()-t0
        print "\t\t\tTime Taken for last MCS:",time.clock()-SIMULATION_TIME
        print "\t\t\tTime Taken (cumulative):",time.clock()-SIMULATION_TIME0,"\n"
        SIMULATION_TIME=time.clock()

#*************************************************************************************************************
class ExecuteClusterMitosis(MitosisSteppableClustersBase):
    def __init__(self,_simulator,_frequency=20):
        MitosisSteppableClustersBase.__init__(self,_simulator,_frequency)
        #SteppableBasePy.__init__(self,_simulator,_frequency)
        #self.mitosisSteppable=CompuCell.MitosisSteppable()
        #self.mitosisSteppable.init(_simulator)
        self.useDivideClusterFunctions = True
        self.divideClusterBaseCellOnly = False
        self.doNotDivideTypeList = [MEDIUM,WALL,MATRIX,BM,TJ,TJm]
        self.clusterBaseTypeList = [MESENCH_N,MESENCH_O,MESENCH_EMT,MATRIX,BM,EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME]
        self.verbose = False
    
    def start(self):
        self.fourthordrxy=[
            [0.0,8.0*sqrt(3)/2], [0.0,-8.0*sqrt(3)/2], [12/2,0],             [-12.0/2,0],
            [3.0/2,7*sqrt(3)/2], [-3.0/2,7*sqrt(3)/2], [3.0/2,-7*sqrt(3)/2], [-3.0/2,-7*sqrt(3)/2],
            [6.0/2, 6*sqrt(3)/2],[-6.0/2,6*sqrt(3)/2], [6.0/2,-6*sqrt(3)/2], [-6.0/2,-6*sqrt(3)/2],
            [9.0/2, 5*sqrt(3)/2],[-9.0/2,5*sqrt(3)/2], [9.0/2,-5*sqrt(3)/2], [-9.0/2,-5*sqrt(3)/2],
            [12.0/2,4*sqrt(3)/2],[-12.0/2,4*sqrt(3)/2],[12.0/2,-4*sqrt(3)/2],[-12.0/2,-4*sqrt(3)/2],
            [12.0/2,2*sqrt(3)/2],[-12.0/2,2*sqrt(3)/2],[12.0/2,-2*sqrt(3)/2],[-12.0/2,-2*sqrt(3)/2]]
    
    def step(self,mcs):
        t0=time.clock()
        print "\n***** ExecuteClusterMitosis (step) **************************"
        # some cells are compartmentalized, others are not, try to make all calcs relevant for both
        global MITOSIS_COUNT_LIST
        orientedClusterMitosisList=[]
        randomClusterMitosisList=[]
        orientedCellMitosisList=[]   # these latter two are because the divideCluster functions are not working on hexagonal lattices
        randomCellMitosisList=[]
        for compartmentList in self.clusterList:
            clusterMitosisFlag = True
            clusterBaseType = 0
            clusterId = 0
            clusterBaseCell = [cell for cell in CompartmentList(compartmentList)][0] #not exactly shorthand to get 1st cell in cluster
            for cell in CompartmentList(compartmentList):
                if cell.type in self.clusterBaseTypeList:
                    clusterBaseType = cell.type
                    clusterBaseCell = cell
                clusterId = cell.clusterId
                cellAttributes=CompuCell.getPyAttrib(cell)
                clusterMitosisFlag = clusterMitosisFlag and (cellAttributes['cell.mitosisFlag'])
            
            if clusterMitosisFlag and (clusterBaseType in [MESENCH_N,MESENCH_O,MESENCH_EMT]):
                #print "\t*** ADDING TO RANDOM MITOSIS LIST ***"
                randomClusterMitosisList.append(clusterId)
                if self.divideClusterBaseCellOnly:
                    randomCellMitosisList.append(clusterBaseCell)
                else:
                    for cell in CompartmentList(compartmentList):
                        if not (cell.type in self.doNotDivideTypeList):
                            randomCellMitosisList.append(cell)
            elif clusterMitosisFlag and (clusterBaseType in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME]):
                #print "\t*** ADDING TO ORIENTED MITOSIS LIST ***"
                orientedClusterMitosisList.append(clusterId)
                if self.divideClusterBaseCellOnly:
                    orientedCellMitosisList.append(clusterBaseCell)
                else:
                    for cell in CompartmentList(compartmentList):
                        if not (cell.type in self.doNotDivideTypeList):
                            orientedCellMitosisList.append(cell)
        print "\t*** randomClusterMitosisList (clusterIds):",randomClusterMitosisList
        for cell in randomCellMitosisList:
            print "\t\tcell",cell.id,"of type",typeList[cell.type],"from clusterId",cell.clusterId
        print "\t*** orientedClusterMitosisList (clusterIds):",orientedClusterMitosisList
        for cell in orientedCellMitosisList:
            print "\t\tcell",cell.id,"of type",typeList[cell.type],"from clusterId",cell.clusterId
    
        if self.useDivideClusterFunctions:
            for clusterId in randomClusterMitosisList:
                #print "Next step is divideClusterRandomOrientation"
                #print "Next step is divideClusterAlongMajorAxis"
                #if self.divideClusterAlongMajorAxis(clusterId):
                if self.divideClusterRandomOrientation(clusterId):
                    #self.updateAttributes()
                    #print "Completed divideClusterRandomOrientation; now update MITOSIS_COUNT_LIST"
                    compartmentList = self.inventory.getClusterCells(clusterId)
                    for i in xrange(compartmentList.size()):
                        #print "   updating for cell.id=",compartmentList[i].id
                        MITOSIS_COUNT_LIST[compartmentList[i].type] +=1
                        if self.verbose: print "Mitosis (random) of cell type=",typeList[compartmentList[i].type]
                else: print "WARNING: Unable to complete RANDOM mitosis of cluster",clusterId
            # oriented mitosis needs to be along MINOR axis so elongated cells become shorter after division
            for clusterId in orientedClusterMitosisList:
                #print "Next step is divideClusterAlongMinorAxis"
                if self.divideClusterAlongMinorAxis(clusterId):
                    #print "Completed divideClusterAlongMinorAxis; now update MITOSIS_COUNT_LIST"
                    compartmentList = self.inventory.getClusterCells(clusterId)
                    for i in xrange(compartmentList.size()):
                        #print "   updating for cell.id=",compartmentList[i].id
                        MITOSIS_COUNT_LIST[compartmentList[i].type] +=1
                        if self.verbose: print "Mitosis (oriented) of cell type=",typeList[compartmentList[i].type]
                else: print "WARNING: Unable to complete ORIENTED mitosis of cluster",clusterId
        else:       # section to run if not using divideCluster functions (i.e., on hex lattice)
            for dividecell in randomCellMitosisList:
                if self.mitosisSteppable.doDirectionalMitosisRandomOrientation(dividecell):
                    self.updateAttributes()
                    MITOSIS_COUNT_LIST[dividecell.type] +=1
                    if self.verbose: print "Mitosis (random) of cell type=",typeList[dividecell.type]
                else: print "WARNING: Unable to complete RANDOM mitosis of cell",dividecell.id,"of type",typeList[dividecell.type]
            # oriented mitosis needs to be along MINOR axis so elongated cells become shorter after division
            for dividecell in orientedCellMitosisList:
                if self.mitosisSteppable.doDirectionalMitosisAlongMinorAxis(dividecell):
                    self.updateAttributes()
                    MITOSIS_COUNT_LIST[dividecell.type] +=1
                    if self.verbose: print "Mitosis (oriented) of cell type=",typeList[dividecell.type]
                else: print "WARNING: Unable to complete ORIENTED mitosis of cell",dividecell.id,"of type",typeList[dividecell.type]
            # need to finish here by reassigning clusterId flags for all the new child cells
            if not self.divideClusterBaseCellOnly: self.reassignChildCellClusterIds()
                    
        print "Cell Type \t Mitosis Count (cumulative)"
        print "----------\t --------------------------"
        for i in range(nCellTypes):
            print "%(type)9s\t%(count)9d" % {'type':typeList[i], 'count':MITOSIS_COUNT_LIST[i]}
                        #print typeList[i]," \t",MITOSIS_COUNT_LIST[i]
        print "\t\t\tTime Taken=", time.clock()-t0,"\n"
    
    def reassignChildCellClusterIds(self):
        needsClusterReassignmentList=[]
        parentClusterIdList=[]
        for cell in self.cellList:
            cell_attributes=CompuCell.getPyAttrib(cell)
            if cell_attributes.has_key('cell.needsClusterReassignmentFlag') and cell_attributes['cell.needsClusterReassignmentFlag']:
                if cell_attributes.has_key('cell.parentClusterId'):
                    needsClusterReassignmentList.append(cell)
                    parentClusterId=cell_attributes['cell.parentClusterId']
                    if not (parentClusterId in parentClusterIdList): parentClusterIdList.append(parentClusterId)
        #print "\n*** reassignChildCellClusterIds ***"
        #print "\t",len(needsClusterReassignmentList),"cells from",len(parentClusterIdList),"parent clusters need cluster reassignment\n"
        for parentClusterId in parentClusterIdList:
            # find all cells in the needsClusterReassignmentList with this parentClusterId
            matchingCellList=[]
            for cell in needsClusterReassignmentList:
                if CompuCell.getPyAttrib(cell)['cell.parentClusterId']==parentClusterId: matchingCellList.append(cell)            
            if len(matchingCellList)>1:
                # find base cell in child and parent clusters
                parentCompartmentList=self.inventory.getClusterCells(parentClusterId)
                childBaseCell=0
                parentBaseCell=0
                #print "Find childBaseCell"
                for cell in matchingCellList:
                    if cell.type in self.clusterBaseTypeList:
                        childBaseCell = cell
                        break
                #print "Find parentBaseCell"
                for i in range(parentCompartmentList.size()):
                    if parentCompartmentList[i].type in self.clusterBaseTypeList:
                        parentBaseCell = parentCompartmentList[i]
                        break
                #print "attempt reassignments"
                if childBaseCell and parentBaseCell:
                    #print "via childBaseCell and parentBaseCell route"
                    CompuCell.getPyAttrib(childBaseCell)['cell.needsClusterReassignmentFlag']=False   # reset flag
                    newClusterId=childBaseCell.clusterId
                    for cell in matchingCellList:
                        if cell.id != childBaseCell.id:
                            dChild = self.getCellCellDistance(childBaseCell,cell)
                            dParent = self.getCellCellDistance(parentBaseCell,cell)
                            #print "\tcentroid distance to childBaseCell=",dChild
                            #print "\tcentroid distance to parentBaseCell=",dParent
                            if dChild>0 and dChild < dParent:
                                self.inventory.reassignClusterId(cell, newClusterId)  # set clusterId to match childBaseCell
                            else:
                                self.inventory.reassignClusterId(cell, parentClusterId)  # set clusterId to match parentBaseCell
                            CompuCell.getPyAttrib(cell)['cell.needsClusterReassignmentFlag']=False   # reset flag
                    for i in range(parentCompartmentList.size()):
                        if parentCompartmentList[i].id != parentBaseCell.id:
                            dChild = self.getCellCellDistance(childBaseCell,parentCompartmentList[i])
                            dParent = self.getCellCellDistance(parentBaseCell,parentCompartmentList[i])
                            #print "\tcentroid distance to childBaseCell=",dChild
                            #print "\tcentroid distance to parentBaseCell=",dParent
                            if dChild>0 and dChild < dParent:    # set clusterId to match childBaseCell
                                self.inventory.reassignClusterId(parentCompartmentList[i], newClusterId)  
                            else:                                # set clusterId to match parentBaseCell
                                self.inventory.reassignClusterId(parentCompartmentList[i], parentClusterId)  
                else:
                    #print "via first cell in matchingCellList"
                    CompuCell.getPyAttrib(matchingCellList[0])['cell.needsClusterReassignmentFlag']=False # reset flag for first cell
                    newClusterId = matchingCellList[0].clusterId   # using first entry as source of newClusterId
                    for cell in matchingCellList[1:]:   
                        self.inventory.reassignClusterId(cell, matchingCellList[0].clusterId)  # set clusterId to match first cell
                        #print "\tcentroid distance between compartments=",self.getCellCellDistance(matchingCellList[0],cell)
                        CompuCell.getPyAttrib(cell)['cell.needsClusterReassignmentFlag']=False   # reset flag
            elif matchingCellList[0]:
                CompuCell.getPyAttrib(matchingCellList[0])['cell.needsClusterReassignmentFlag']=False # reset flag for only cell
    

    def getCellCellDistance(self, _cell1, _cell2):
        if _cell1.volume>0 and _cell2.volume>0:
            dx = _cell1.xCM/float(_cell1.volume) - _cell2.xCM/float(_cell2.volume)
            dy = _cell1.yCM/float(_cell1.volume) - _cell2.yCM/float(_cell2.volume)
            dz = _cell1.zCM/float(_cell1.volume) - _cell2.zCM/float(_cell2.volume)
            return sqrt(dx*dx + dy*dy + dz*dz)
        else:
            return 0.0

    def updateAttributes(self):
        childCell=self.mitosisSteppable.childCell
        parentCell=self.mitosisSteppable.parentCell
        randommultiplier=random.uniform(.465,.535)
        if self.useDivideClusterFunctions:
            compartmentListChild  = self.inventory.getClusterCells(childCell.clusterId)
            compartmentListParent = self.inventory.getClusterCells(parentCell.clusterId)
            for i in xrange(compartmentListChild.size()):
                # parameters set for all types of cells
                compartmentListChild[i].lambdaVolume= compartmentListParent[i].lambdaVolume
                compartmentListChild[i].lambdaSurface=compartmentListParent[i].lambdaSurface
                # parameters whose settings depend on cell type
                compartmentType = compartmentListChild[i].type
                if compartmentType in [TJ,TJm]:
                    compartmentListChild[i].targetVolume= compartmentListParent[i].targetVolume
                    compartmentListChild[i].targetSurface=compartmentListParent[i].targetSurface
                else:
                    compartmentListChild[i].targetVolume=randommultiplier*compartmentListParent[i].targetVolume
                    compartmentListParent[i].targetVolume=(1-randommultiplier)*compartmentListParent[i].targetVolume
                    if compartmentType in [MESENCH_N,MESENCH_O,MESENCH_EMT,EPI_N,EPI_O,EPI_ME]:  #cuboidal cells
                        compartmentListParent[i].targetSurface=4*sqrt(max(0,compartmentListParent[i].targetVolume))
                        compartmentListChild[i].targetSurface= 4*sqrt(max(0,compartmentListChild[i].targetVolume))
                    elif compartmentType in [PERI_N,PERI_O,PERI_ME]:    # select targets for flattened unpolarized cells; adjusted in next CellBehavior
                        compartmentListParent[i].targetSurface=0.7*(compartmentListParent[i].targetVolume)
                        compartmentListChild[i].targetSurface= 0.7*(compartmentListChild[i].targetVolume)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListParent[i], 20*compartmentListParent[i].lambdaSurface,  compartmentListParent[i].targetVolume*0.3)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListChild[i], 20*compartmentListChild[i].lambdaSurface,  compartmentListChild[i].targetVolume*0.3)
                    elif compartmentType in [PERI_Na,PERI_Oa,PERI_MEa]:    # very flattened cells
                        compartmentListParent[i].targetSurface=1.0*(compartmentListParent[i].targetVolume)
                        compartmentListChild[i].targetSurface= 1.0*(compartmentListChild[i].targetVolume)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListParent[i], 20*compartmentListParent[i].lambdaSurface,  compartmentListParent[i].targetVolume*0.5)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListChild[i], 20*compartmentListChild[i].lambdaSurface,  compartmentListChild[i].targetVolume*0.5)
 
                parent_attributes=CompuCell.getPyAttrib(compartmentListParent[i])
                child_attributes=CompuCell.getPyAttrib(compartmentListChild[i])
                parent_attributes['cell.generation']+=1
                #parent_attributes['cell.pressureCount']=0
                if parentCell.type in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME,PERI_Na,PERI_Oa,PERI_MEa]:
                    parent_attributes['cell.timer.ectoDivision']=50
                child_attributes.update(parent_attributes)
                child_attributes['cell.parent']=compartmentListParent[i].id
		parent_attributes['cell.mitosisFlag']=False
		child_attributes['cell.mitosisFlag']=False
                parent_attributes['cell.timer.mitosis']=0    #counts up from last mitosis
		child_attributes['cell.timer.mitosis']=0
        
        else:   # section to run if not using divideCluster functions (i.e., on hex lattice)
            childCell.targetVolume=randommultiplier*parentCell.targetVolume
            parentCell.targetVolume=(1-randommultiplier)*parentCell.targetVolume
            if parentCell.type in [MESENCH_N,MESENCH_O,MESENCH_EMT,EPI_N,EPI_O,EPI_ME]:  #cuboidal cells
                parentCell.targetSurface=4*sqrt(max(0,parentCell.targetVolume))
                childCell.targetSurface=4*sqrt(max(0,childCell.targetVolume))
            elif parentCell.type in [PERI_N,PERI_O,PERI_ME]:                         #flattened cells
                parentCell.targetSurface=0.7*parentCell.targetVolume
                childCell.targetSurface=0.7*childCell.targetVolume
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (parentCell, 20*parentCell.lambdaSurface, parentCell.targetVolume*0.3)
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (childCell, 20*parentCell.lambdaSurface, childCell.targetVolume*0.3)
            elif parentCell.type in [PERI_Na,PERI_Oa,PERI_MEa]:    # very flattened cells
                parentCell.targetSurface=1.0*parentCell.targetVolume
                childCell.targetSurface=1.0*childCell.targetVolume
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (parentCell, 20*parentCell.lambdaSurface, parentCell.targetVolume*0.5)
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (childCell, 20*parentCell.lambdaSurface, childCell.targetVolume*0.5)
            childCell.lambdaVolume=parentCell.lambdaVolume
            childCell.lambdaSurface=parentCell.lambdaSurface
            childCell.type=parentCell.type

            parent_attributes=CompuCell.getPyAttrib(parentCell)
            child_attributes=CompuCell.getPyAttrib(childCell)
            parent_attributes['cell.generation']+=1
            #parent_attributes['cell.pressureCount']=0
            if parentCell.type in [EPI_N,EPI_O,EPI_ME,PERI_N,PERI_O,PERI_ME]:
                parent_attributes['cell.timer.ectoDivision']=50
            child_attributes.update(parent_attributes)
            child_attributes['cell.parent']=parentCell.id
            child_attributes['cell.parentClusterId']=parentCell.clusterId
            child_attributes['cell.needsClusterReassignmentFlag']=True
	    parent_attributes['cell.mitosisFlag']=False
	    child_attributes['cell.mitosisFlag']=False
            parent_attributes['cell.timer.mitosis']=0    #counts up from last mitosis
            child_attributes['cell.timer.mitosis']=0


#*************************************************************************************************************
# This steppable is not currently in use, so I didn't update it to handle "_O" cell types
class PalateModelMitosis(SteppableBasePy):
    def __init__(self,_simulator,_frequency=20):
        SteppableBasePy.__init__(self,_simulator,_frequency)
        self.mitosisSteppable=CompuCell.MitosisSteppable()
        self.mitosisSteppable.init(_simulator)
        self.verbose = False
    
    def start(self):
        self.fourthordrxy=[[0.0,8.0*sqrt(3)/2],[0.0,-8.0*sqrt(3)/2],[12/2,0],[-12.0/2,0],[3.0/2,7*sqrt(3)/2],[-3.0/2,7*sqrt(3)/2],[3.0/2,-7*sqrt(3)/2],[-3.0/2,-7*sqrt(3)/2],[6.0/2, 6*sqrt(3)/2],[-6.0/2,6*sqrt(3)/2],[6.0/2,-6*sqrt(3)/2],[-6.0/2,-6*sqrt(3)/2],[9.0/2, 5*sqrt(3)/2],[-9.0/2,5*sqrt(3)/2],[9.0/2,-5*sqrt(3)/2],[-9.0/2,-5*sqrt(3)/2],[12.0/2,4*sqrt(3)/2],[-12.0/2,4*sqrt(3)/2],[12.0/2,-4*sqrt(3)/2],[-12.0/2,-4*sqrt(3)/2],[12.0/2,2*sqrt(3)/2],[-12.0/2,2*sqrt(3)/2],[12.0/2,-2*sqrt(3)/2],[-12.0/2,-2*sqrt(3)/2]]
    
    def step(self,mcs):
        global MITOSIS_COUNT_LIST
        x=time.clock()
        orientedcelldivisionlist=[]
        randomcelldivisionlist=[]
        for cell in self.cellList:
            if (cell.type==MESENCH_N or cell.type==MESENCH_EMT) and cell.volume>max(MITOSIS_VOLUME_LIST[MESENCH_N:MESENCH_EMT]):
                randomcelldivisionlist.append(cell)
            cellAttributes=CompuCell.getPyAttrib(cell)
            if cell.type in [EPI_N,EPI_ME,PERI_N,PERI_ME] and cellAttributes['cell.timer.ectoDivision']==0:
                print "EPI or PERI cell with ectoDivision timer==0"
                if cell.volume>MITOSIS_VOLUME_LIST[cell.type]:
                    print "EPI or PERI cell with sufficient volume for mitosis"
                    aspect = self.momentOfInertiaPlugin.getSemiaxes(cell)[2]/self.momentOfInertiaPlugin.getSemiaxes(cell)[0]
                    if   cell.type==EPI_N  and aspect>4.0: orientedcelldivisionlist.append(cell)
                    elif cell.type==EPI_ME  and aspect>2.0: orientedcelldivisionlist.append(cell)
                    elif cell.type==PERI_N and aspect>4.0: orientedcelldivisionlist.append(cell)
                    elif cell.type==PERI_ME and aspect>3.0: orientedcelldivisionlist.append(cell)
        
        
        #[self.updateAttributes() for dividecell in randomcelldivisionlist if self.mitosisSteppable.doDirectionalMitosisRandomOrientation(dividecell)]
        for dividecell in randomcelldivisionlist:
            if self.mitosisSteppable.doDirectionalMitosisRandomOrientation(dividecell):
                self.updateAttributes()
                MITOSIS_COUNT_LIST[dividecell.type] +=1
                if self.verbose: print "Mitosis (random) of cell type=",typeList[dividecell.type]
            else: print "WARNING: Unable to complete mitosis of cell",dividecell.id,"of type",typeList[dividecell.type]
        # oriented mitosis needs to be along MINOR axis so elongated cells become shorter after division
        for dividecell in orientedcelldivisionlist:
            if self.mitosisSteppable.doDirectionalMitosisAlongMinorAxis(dividecell):
                self.updateAttributes()
                MITOSIS_COUNT_LIST[dividecell.type] +=1
                if self.verbose: print "Mitosis (oriented) of cell type=",typeList[dividecell.type]
            else: print "WARNING: Unable to complete mitosis of cell",dividecell.id,"of type",typeList[dividecell.type]
        print "Cell Type \t Mitosis Count (cumulative)"
        for i in range(nCellTypes):
            print "%(type)9s\t%(count)9d" % {'type':typeList[i], 'count':MITOSIS_COUNT_LIST[i]}
        #print typeList[i]," \t",MITOSIS_COUNT_LIST[i]
        print "Mitosis,              Time Taken=", time.clock()-x
    
    def updateAttributes(self):
        childCell=self.mitosisSteppable.childCell
        parentCell=self.mitosisSteppable.parentCell
        parent_attributes=CompuCell.getPyAttrib(parentCell)
        child_attributes=CompuCell.getPyAttrib(childCell)
        
        randommultiplier=random.uniform(.465,.535)
        childCell.targetVolume=randommultiplier*parentCell.targetVolume
        parentCell.targetVolume=(1-randommultiplier)*parentCell.targetVolume
        
        if parentCell.type in [PERI_N,PERI_ME]:                         #flattened cells
            parentCell.targetSurface=0.7*parentCell.targetVolume
            childCell.targetSurface=0.7*childCell.targetVolume
            self.lengthConstraintPlugin.setLengthConstraintData \
                (parentCell, 20*parentCell.lambdaSurface, parentCell.targetVolume*0.3)
            self.lengthConstraintPlugin.setLengthConstraintData \
                (childCell, 20*parentCell.lambdaSurface, childCell.targetVolume*0.3)
        else:                                                       #cuboidal cells
            parentCell.targetSurface=4*sqrt(max(0,parentCell.targetVolume))
            childCell.targetSurface=4*sqrt(max(0,childCell.targetVolume))
        childCell.lambdaVolume=parentCell.lambdaVolume
        childCell.lambdaSurface=parentCell.lambdaSurface
        childCell.type=parentCell.type

        parent_attributes['cell.generation']+=1
        #parent_attributes['cell.pressureCount']=0

        if parentCell.type in [EPI_N,EPI_ME,PERI_N,PERI_ME]:
            parent_attributes['cell.timer.ectoDivision']=50

        child_attributes.update(parent_attributes)
        child_attributes['cell.parent']=parentCell.id

#if parentCell.type==2 and parent_attributes['cell.toAERFlag']==True:
#    parent_attributes['cell.toAERFlag']=child_attributes['cell.toAERFlag']=False
#    if random.random()<.5:
#        parentCell.type=3
#        childCell.type=1
#    else:
#        parentCell.type=1
#        childCell.type=3



#*************************************************************************************************************
# This steppable is not currently in use, so I didn't update it to handle "_O" cell types
class PalateModelClusterMitosis(MitosisSteppableClustersBase):
    def __init__(self,_simulator,_frequency=20):
        MitosisSteppableClustersBase.__init__(self,_simulator,_frequency)
        #SteppableBasePy.__init__(self,_simulator,_frequency)
        #self.mitosisSteppable=CompuCell.MitosisSteppable()
        #self.mitosisSteppable.init(_simulator)
        self.useDivideClusterFunctions = True
        self.divideClusterBaseCellOnly = False
        self.doNotDivideTypeList = [WALL,TJ,TJm]
        self.clusterBaseTypeList = [MESENCH_N,MESENCH_EMT,EPI_N,EPI_ME,PERI_N,PERI_ME]
        self.verbose = False
    
    def start(self):
        self.fourthordrxy=[[0.0,8.0*sqrt(3)/2],[0.0,-8.0*sqrt(3)/2],[12/2,0],[-12.0/2,0],[3.0/2,7*sqrt(3)/2],[-3.0/2,7*sqrt(3)/2],[3.0/2,-7*sqrt(3)/2],[-3.0/2,-7*sqrt(3)/2],[6.0/2, 6*sqrt(3)/2],[-6.0/2,6*sqrt(3)/2],[6.0/2,-6*sqrt(3)/2],[-6.0/2,-6*sqrt(3)/2],[9.0/2, 5*sqrt(3)/2],[-9.0/2,5*sqrt(3)/2],[9.0/2,-5*sqrt(3)/2],[-9.0/2,-5*sqrt(3)/2],[12.0/2,4*sqrt(3)/2],[-12.0/2,4*sqrt(3)/2],[12.0/2,-4*sqrt(3)/2],[-12.0/2,-4*sqrt(3)/2],[12.0/2,2*sqrt(3)/2],[-12.0/2,2*sqrt(3)/2],[12.0/2,-2*sqrt(3)/2],[-12.0/2,-2*sqrt(3)/2]]
    
    def step(self,mcs):
        t0=time.clock()
        print "\n***** PalateModelClusterMitosis (step) **************************"
        # some cells are compartmentalized, others are not, try to make all calcs relevant for both
        global MITOSIS_COUNT_LIST
        orientedClusterMitosisList=[]
        randomClusterMitosisList=[]
        orientedCellMitosisList=[]   # these latter two are because the divideCluster functions are not working on hexagonal lattices
        randomCellMitosisList=[]
        for compartmentList in self.clusterList:
            timerFlag = True
            clusterVolume = 0
            clusterBaseType = 0
            clusterId = 0
            maxAspect = 1
            clusterBaseCell = 0
            for cell in CompartmentList(compartmentList):
                if cell.type in self.clusterBaseTypeList:
                    clusterBaseType = cell.type
                    clusterBaseCell = cell
                clusterVolume += cell.volume
                clusterId = cell.clusterId
                cellAttributes=CompuCell.getPyAttrib(cell)
                timerFlag = timerFlag and (cellAttributes['cell.timer.ectoDivision']==0) #false if timer !=0 for any cell in cluster
                if self.momentOfInertiaPlugin.getSemiaxes(cell)[0]>0:
                    aspect = self.momentOfInertiaPlugin.getSemiaxes(cell)[2]/self.momentOfInertiaPlugin.getSemiaxes(cell)[0]
                    if aspect > maxAspect: maxAspect = aspect
            volumeFlag = clusterVolume > MITOSIS_VOLUME_LIST[clusterBaseType]
            aspectFlag = False
            if   clusterBaseType==EPI_N  and aspect>3.0: aspectFlag=True
            elif clusterBaseType==EPI_ME  and aspect>2.0: aspectFlag=True
            elif clusterBaseType==PERI_N and aspect>5.0: aspectFlag=True
            elif clusterBaseType==PERI_ME and aspect>3.0: aspectFlag=True
            #print "ClusterId=",clusterId," clusterBaseType=",typeList[clusterBaseType],"clusterVolume=",clusterVolume,"maxAspect=",maxAspect
            #print "\tFLAGS (volume, timer, aspect)=", volumeFlag, timerFlag, aspectFlag
            
            if (clusterBaseType in [MESENCH_N,MESENCH_O,MESENCH_EMT]) and timerFlag and volumeFlag:
                #print "\t*** ADDING TO RANDOM MITOSIS LIST ***"
                randomClusterMitosisList.append(clusterId)
                if self.divideClusterBaseCellOnly:
                    randomCellMitosisList.append(clusterBaseCell)
                else:
                    for cell in CompartmentList(compartmentList):
                        if not (cell.type in self.doNotDivideTypeList):
                            randomCellMitosisList.append(cell)
            elif (clusterBaseType in [EPI_N,EPI_ME,PERI_N,PERI_ME]) and timerFlag and volumeFlag and aspectFlag:
                #print "\t*** ADDING TO ORIENTED MITOSIS LIST ***"
                orientedClusterMitosisList.append(clusterId)
                if self.divideClusterBaseCellOnly:
                    orientedCellMitosisList.append(clusterBaseCell)
                else:
                    for cell in CompartmentList(compartmentList):
                        if not (cell.type in self.doNotDivideTypeList):
                            orientedCellMitosisList.append(cell)
        print "\t*** randomClusterMitosisList (clusterIds):",randomClusterMitosisList
        for cell in randomCellMitosisList:
            print "\t\tcell",cell.id,"of type",typeList[cell.type],"from clusterId",cell.clusterId
        print "\t*** orientedClusterMitosisList (clusterIds):",orientedClusterMitosisList
        for cell in orientedCellMitosisList:
            print "\t\tcell",cell.id,"of type",typeList[cell.type],"from clusterId",cell.clusterId
    
        if self.useDivideClusterFunctions:
            for clusterId in randomClusterMitosisList:
                #print "Next step is divideClusterRandomOrientation"
                #print "Next step is divideClusterAlongMajorAxis"
                #if self.divideClusterAlongMajorAxis(clusterId):
                if self.divideClusterRandomOrientation(clusterId):
                    #self.updateAttributes()
                    #print "Completed divideClusterRandomOrientation; now update MITOSIS_COUNT_LIST"
                    compartmentList = self.inventory.getClusterCells(clusterId)
                    for i in xrange(compartmentList.size()):
                        #print "   updating for cell.id=",compartmentList[i].id
                        MITOSIS_COUNT_LIST[compartmentList[i].type] +=1
                        if self.verbose: print "Mitosis (random) of cell type=",typeList[compartmentList[i].type]
                else: print "WARNING: Unable to complete RANDOM mitosis of cluster",clusterId
            # oriented mitosis needs to be along MINOR axis so elongated cells become shorter after division
            for clusterId in orientedClusterMitosisList:
                #print "Next step is divideClusterAlongMinorAxis"
                if self.divideClusterAlongMinorAxis(clusterId):
                    #print "Completed divideClusterAlongMinorAxis; now update MITOSIS_COUNT_LIST"
                    compartmentList = self.inventory.getClusterCells(clusterId)
                    for i in xrange(compartmentList.size()):
                        #print "   updating for cell.id=",compartmentList[i].id
                        MITOSIS_COUNT_LIST[compartmentList[i].type] +=1
                        if self.verbose: print "Mitosis (oriented) of cell type=",typeList[compartmentList[i].type]
                else: print "WARNING: Unable to complete ORIENTED mitosis of cluster",clusterId
        else:       # section to run if not using divideCluster functions (i.e., on hex lattice)
            for dividecell in randomCellMitosisList:
                if self.mitosisSteppable.doDirectionalMitosisRandomOrientation(dividecell):
                    self.updateAttributes()
                    MITOSIS_COUNT_LIST[dividecell.type] +=1
                    if self.verbose: print "Mitosis (random) of cell type=",typeList[dividecell.type]
                else: print "WARNING: Unable to complete RANDOM mitosis of cell",dividecell.id,"of type",typeList[dividecell.type]
            # oriented mitosis needs to be along MINOR axis so elongated cells become shorter after division
            for dividecell in orientedCellMitosisList:
                if self.mitosisSteppable.doDirectionalMitosisAlongMinorAxis(dividecell):
                    self.updateAttributes()
                    MITOSIS_COUNT_LIST[dividecell.type] +=1
                    if self.verbose: print "Mitosis (oriented) of cell type=",typeList[dividecell.type]
                else: print "WARNING: Unable to complete ORIENTED mitosis of cell",dividecell.id,"of type",typeList[dividecell.type]
            # need to finish here by reassigning clusterId flags for all the new child cells
            if not self.divideClusterBaseCellOnly: self.reassignChildCellClusterIds()
                    
        print "Cell Type \t Mitosis Count (cumulative)"
        print "----------\t --------------------------"
        for i in range(nCellTypes):
            print "%(type)9s\t%(count)9d" % {'type':typeList[i], 'count':MITOSIS_COUNT_LIST[i]}
                        #print typeList[i]," \t",MITOSIS_COUNT_LIST[i]
        print "\t\t\tTime Taken=", time.clock()-t0,"\n"
    
    def reassignChildCellClusterIds(self):
        needsClusterReassignmentList=[]
        parentClusterIdList=[]
        for cell in self.cellList:
            cell_attributes=CompuCell.getPyAttrib(cell)
            if cell_attributes.has_key('cell.needsClusterReassignmentFlag') and cell_attributes['cell.needsClusterReassignmentFlag']:
                if cell_attributes.has_key('cell.parentClusterId'):
                    needsClusterReassignmentList.append(cell)
                    parentClusterId=cell_attributes['cell.parentClusterId']
                    if not (parentClusterId in parentClusterIdList): parentClusterIdList.append(parentClusterId)
        #print "\n*** reassignChildCellClusterIds ***"
        #print "\t",len(needsClusterReassignmentList),"cells from",len(parentClusterIdList),"parent clusters need cluster reassignment\n"
        for parentClusterId in parentClusterIdList:
            # find all cells in the needsClusterReassignmentList with this parentClusterId
            matchingCellList=[]
            for cell in needsClusterReassignmentList:
                if CompuCell.getPyAttrib(cell)['cell.parentClusterId']==parentClusterId: matchingCellList.append(cell)            
            if len(matchingCellList)>1:
                # find base cell in child and parent clusters
                parentCompartmentList=self.inventory.getClusterCells(parentClusterId)
                childBaseCell=0
                parentBaseCell=0
                #print "Find childBaseCell"
                for cell in matchingCellList:
                    if cell.type in self.clusterBaseTypeList:
                        childBaseCell = cell
                        break
                #print "Find parentBaseCell"
                for i in range(parentCompartmentList.size()):
                    if parentCompartmentList[i].type in self.clusterBaseTypeList:
                        parentBaseCell = parentCompartmentList[i]
                        break
                #print "attempt reassignments"
                if childBaseCell and parentBaseCell:
                    #print "via childBaseCell and parentBaseCell route"
                    CompuCell.getPyAttrib(childBaseCell)['cell.needsClusterReassignmentFlag']=False   # reset flag
                    newClusterId=childBaseCell.clusterId
                    for cell in matchingCellList:
                        if cell.id != childBaseCell.id:
                            dChild = self.getCellCellDistance(childBaseCell,cell)
                            dParent = self.getCellCellDistance(parentBaseCell,cell)
                            #print "\tcentroid distance to childBaseCell=",dChild
                            #print "\tcentroid distance to parentBaseCell=",dParent
                            if dChild>0 and dChild < dParent:
                                self.inventory.reassignClusterId(cell, newClusterId)  # set clusterId to match childBaseCell
                            else:
                                self.inventory.reassignClusterId(cell, parentClusterId)  # set clusterId to match parentBaseCell
                            CompuCell.getPyAttrib(cell)['cell.needsClusterReassignmentFlag']=False   # reset flag
                    for i in range(parentCompartmentList.size()):
                        if parentCompartmentList[i].id != parentBaseCell.id:
                            dChild = self.getCellCellDistance(childBaseCell,parentCompartmentList[i])
                            dParent = self.getCellCellDistance(parentBaseCell,parentCompartmentList[i])
                            #print "\tcentroid distance to childBaseCell=",dChild
                            #print "\tcentroid distance to parentBaseCell=",dParent
                            if dChild>0 and dChild < dParent:    # set clusterId to match childBaseCell
                                self.inventory.reassignClusterId(parentCompartmentList[i], newClusterId)  
                            else:                                # set clusterId to match parentBaseCell
                                self.inventory.reassignClusterId(parentCompartmentList[i], parentClusterId)  
                else:
                    #print "via first cell in matchingCellList"
                    CompuCell.getPyAttrib(matchingCellList[0])['cell.needsClusterReassignmentFlag']=False # reset flag for first cell
                    newClusterId = matchingCellList[0].clusterId   # using first entry as source of newClusterId
                    for cell in matchingCellList[1:]:   
                        self.inventory.reassignClusterId(cell, matchingCellList[0].clusterId)  # set clusterId to match first cell
                        #print "\tcentroid distance between compartments=",self.getCellCellDistance(matchingCellList[0],cell)
                        CompuCell.getPyAttrib(cell)['cell.needsClusterReassignmentFlag']=False   # reset flag
            elif matchingCellList[0]:
                CompuCell.getPyAttrib(matchingCellList[0])['cell.needsClusterReassignmentFlag']=False # reset flag for only cell
    

    def getCellCellDistance(self, _cell1, _cell2):
        if _cell1.volume>0 and _cell2.volume>0:
            dx = _cell1.xCM/float(_cell1.volume) - _cell2.xCM/float(_cell2.volume)
            dy = _cell1.yCM/float(_cell1.volume) - _cell2.yCM/float(_cell2.volume)
            dz = _cell1.zCM/float(_cell1.volume) - _cell2.zCM/float(_cell2.volume)
            return sqrt(dx*dx + dy*dy + dz*dz)
        else:
            return 0.0

    def updateAttributes(self):
        childCell=self.mitosisSteppable.childCell
        parentCell=self.mitosisSteppable.parentCell
        randommultiplier=random.uniform(.465,.535)
        if self.useDivideClusterFunctions:
            compartmentListChild  = self.inventory.getClusterCells(childCell.clusterId)
            compartmentListParent = self.inventory.getClusterCells(parentCell.clusterId)
            for i in xrange(compartmentListChild.size()):
                # parameters set for all types of cells
                compartmentListChild[i].lambdaVolume= compartmentListParent[i].lambdaVolume
                compartmentListChild[i].lambdaSurface=compartmentListParent[i].lambdaSurface
                # parameters whose settings depend on cell type
                compartmentType = compartmentListChild[i].type
                if compartmentType in [TJ,TJm]:
                    compartmentListChild[i].targetVolume= compartmentListParent[i].targetVolume
                    compartmentListChild[i].targetSurface=compartmentListParent[i].targetSurface
                else:
                    compartmentListChild[i].targetVolume=randommultiplier*compartmentListParent[i].targetVolume
                    compartmentListParent[i].targetVolume=(1-randommultiplier)*compartmentListParent[i].targetVolume
                    if compartmentType in [MESENCH_N,MESENCH_EMT,EPI_N,EPI_ME]:  #cuboidal cells
                        compartmentListParent[i].targetSurface=4*sqrt(max(0,compartmentListParent[i].targetVolume))
                        compartmentListChild[i].targetSurface= 4*sqrt(max(0,compartmentListChild[i].targetVolume))
                    elif compartmentType in [PERI_N,PERI_ME]:    # select targets for flattened unpolarized cells; adjusted in next CellBehavior
                        compartmentListParent[i].targetSurface=0.7*(compartmentListParent[i].targetVolume)
                        compartmentListChild[i].targetSurface= 0.7*(compartmentListChild[i].targetVolume)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListParent[i], 20*compartmentListParent[i].lambdaSurface,  compartmentListParent[i].targetVolume*0.3)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListChild[i], 20*compartmentListChild[i].lambdaSurface,  compartmentListChild[i].targetVolume*0.3)
                    elif compartmentType in [PERI_Na,PERI_MEa]:    # very flattened cells
                        compartmentListParent[i].targetSurface=1.0*(compartmentListParent[i].targetVolume)
                        compartmentListChild[i].targetSurface= 1.0*(compartmentListChild[i].targetVolume)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListParent[i], 20*compartmentListParent[i].lambdaSurface,  compartmentListParent[i].targetVolume*0.5)
                        self.lengthConstraintPlugin.setLengthConstraintData( compartmentListChild[i], 20*compartmentListChild[i].lambdaSurface,  compartmentListChild[i].targetVolume*0.5)
 
                parent_attributes=CompuCell.getPyAttrib(compartmentListParent[i])
                child_attributes=CompuCell.getPyAttrib(compartmentListChild[i])
                parent_attributes['cell.generation']+=1
                #parent_attributes['cell.pressureCount']=0
                if parentCell.type in [EPI_N,EPI_ME,PERI_N,PERI_ME,PERI_Na,PERI_MEa]:
                    parent_attributes['cell.timer.ectoDivision']=50
                child_attributes.update(parent_attributes)
                child_attributes['cell.parent']=compartmentListParent[i].id
        
        else:   # section to run if not using divideCluster functions (i.e., on hex lattice)
            childCell.targetVolume=randommultiplier*parentCell.targetVolume
            parentCell.targetVolume=(1-randommultiplier)*parentCell.targetVolume
            if parentCell.type in [MESENCH_N,MESENCH_EMT,EPI_N,EPI_ME]:  #cuboidal cells
                parentCell.targetSurface=4*sqrt(max(0,parentCell.targetVolume))
                childCell.targetSurface=4*sqrt(max(0,childCell.targetVolume))
            elif parentCell.type in [PERI_N,PERI_ME]:                         #flattened cells
                parentCell.targetSurface=0.7*parentCell.targetVolume
                childCell.targetSurface=0.7*childCell.targetVolume
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (parentCell, 20*parentCell.lambdaSurface, parentCell.targetVolume*0.3)
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (childCell, 20*parentCell.lambdaSurface, childCell.targetVolume*0.3)
            elif parentCell.type in [PERI_Na,PERI_MEa]:    # very flattened cells
                parentCell.targetSurface=1.0*parentCell.targetVolume
                childCell.targetSurface=1.0*childCell.targetVolume
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (parentCell, 20*parentCell.lambdaSurface, parentCell.targetVolume*0.5)
                self.lengthConstraintPlugin.setLengthConstraintData \
                    (childCell, 20*parentCell.lambdaSurface, childCell.targetVolume*0.5)
            childCell.lambdaVolume=parentCell.lambdaVolume
            childCell.lambdaSurface=parentCell.lambdaSurface
            childCell.type=parentCell.type

            parent_attributes=CompuCell.getPyAttrib(parentCell)
            child_attributes=CompuCell.getPyAttrib(childCell)
            parent_attributes['cell.generation']+=1
            #parent_attributes['cell.pressureCount']=0
            if parentCell.type in [EPI_N,EPI_ME,PERI_N,PERI_ME]:
                parent_attributes['cell.timer.ectoDivision']=50
            child_attributes.update(parent_attributes)
            child_attributes['cell.parent']=parentCell.id
            child_attributes['cell.parentClusterId']=parentCell.clusterId
            child_attributes['cell.needsClusterReassignmentFlag']=True


#*************************************************************************************************************
class ManageEpithelialPolarization(SteppableBasePy):                        # manages behavior of cells with epitheial polarization
    def __init__(self,_simulator,_frequency=1):                             # including tight junctions between cells
        SteppableBasePy.__init__(self,_simulator,_frequency)                # and creation of special apical domains
        self.Potts = self.simulator.getPotts()
        self.cellFieldG = self.Potts.getCellFieldG()
        # default values
        self.startMCS=0
        self.probNucleate=0
        self.probRevert=0
        self.probDestabilize=0
        self.nucleateFromCellTypeList=[]
        self.nucleateWhenContactingCellTypeList=[]
        self.revertWhenNotContactingCellTypeList=[]
        self.TJTargetVolume = 0
        self.TJLambdaVolume = 0
        self.TJLambdaSurface = 0
        self.transientNucleationMultiplier = 1
        self.lambdaFPP = 10
        self.useSpecialApicalDomain = False
        self.useSpecialJunctionDomain = False
        self.polarizedCompartmentsList=[]   #list of dicts, i.e. [ ['basal':PERI_N, 'apical':PERI_Na],['basal':PERI_ME, 'apical':PERI_MEa] ]
        self.verbose=False
    
    
    def setStartMCS(self, _startMCS):
        self.startMCS = _startMCS
    
    def setTJProbabilities(self, _probNucleate, _probRevert, _probDestabilize):
        self.probNucleate=_probNucleate # in units of probability per cell per mcs
        self.probRevert=_probRevert # in units of probability per cell per mcs
        self.probDestabilize=_probDestabilize # in units of probability per cell per mcs
    
    def setNucleateTJFromCellTypes(self, _nucleateFromCellTypeList):
        self.nucleateFromCellTypeList = _nucleateFromCellTypeList
                    
    def setNucleateTJWhenContactingCellTypes(self, _nucleateWhenContactingCellTypeList):
        self.nucleateWhenContactingCellTypeList = _nucleateWhenContactingCellTypeList
                    
    def setRevertTJWhenNotContactingCellTypes(self, _revertWhenNotContactingCellTypeList):
        self.revertWhenNotContactingCellTypeList = _revertWhenNotContactingCellTypeList
                    
    def setTJNucleationParameters(self, _TJTargetVolume, _TJLambdaVolume, _TJLambdaSurface, _transientNucleationMultiplier, _maxTJperCell):
        self.TJTargetVolume = _TJTargetVolume
        self.TJLambdaVolume = _TJLambdaVolume
        self.TJLambdaSurface = _TJLambdaSurface
        self.transientNucleationMultiplier = _transientNucleationMultiplier
        self.maxTJperCell=_maxTJperCell
    
    def addPolarizedCompartmentSet(self, _apical, _basal):
        polarizedCompartments = {'apical':_apical, 'basal':_basal}
        self.polarizedCompartmentsList.append(polarizedCompartments)

    def getEquivalentCircleSurface(self, _volume):
        return 2.0*(pi*_volume)**(0.5)
            
    def isBoundaryCell(self, _cell, _boundedCellTypeList):
        cellNeighborList=CellNeighborListAuto(self.neighborTrackerPlugin,_cell)
        boundaryCellFlag = False
        for neighborSurfaceData in cellNeighborList:
            if neighborSurfaceData.neighborAddress:
                if neighborSurfaceData.neighborAddress.type in _boundedCellTypeList:
                    boundaryCellFlag=True
            elif 0 in _boundedCellTypeList: #necessary to consider cells that border Medium,
                # only gets to this elif when neighborSurfaceData.neighborAddress doesn't exist, i.e. neighbor is Medium
                boundaryCellFlag = True
        return boundaryCellFlag
    
    def pickSurfacePixel(self, _cell, _boundedCellTypeList, _awayFromPoint):
        if self.verbose: print "Trying to find pt in cell",_cell.id,"that bounds cell of type",_boundedCellTypeList
        pixelList=CellBoundaryPixelList(self.boundaryPixelTrackerPlugin,_cell)
        selectPixelList=[]
        for boundaryPixelTrackerData in pixelList:
            addThisPixel=False
            #if self.verbose: print "pixel of cell id=",_cell.id," type:",_cell.type, " = ",boundaryPixelTrackerData.pixel
            # check which pixels border cell types in _boundedCellTypeList
            pt = boundaryPixelTrackerData.pixel
            pt1 = Point3D(pt.x+1, pt.y, pt.z)
            pt2 = Point3D(pt.x, pt.y+1, pt.z)
            pt3 = Point3D(pt.x-1, pt.y, pt.z)
            pt4 = Point3D(pt.x, pt.y-1, pt.z)
            if pt.x==0:            pt3 = Point3D(self.dim.x-1, pt.y, pt.z)
            if pt.x==self.dim.x-1: pt1 = Point3D(0, pt.y, pt.z)
            if pt.y==0:            pt4 = Point3D(pt.x, self.dim.y-1, pt.z)
            if pt.y==self.dim.y-1: pt2 = Point3D(pt.x, 0, pt.z)
            cell1 = self.cellFieldG.get(pt1)
            cell2 = self.cellFieldG.get(pt2)
            cell3 = self.cellFieldG.get(pt3)
            cell4 = self.cellFieldG.get(pt4)
            for cell in [cell1, cell2, cell3, cell4]:
                if 0 in _boundedCellTypeList and not cell: addThisPixel=True
                elif cell and cell.type in _boundedCellTypeList and cell.clusterId!=_cell.clusterId: addThisPixel=True
            if addThisPixel:
                selectPixelList.append(pt)
        #if self.verbose: print "         at surface with cell.types=",_boundedCellTypeList
        if len(selectPixelList)>=1:      
            if not _awayFromPoint:
                return selectPixelList[random.randint(0,len(selectPixelList)-1)]
            else:
                farthestPoint=selectPixelList[0]
                maxDsquared=0
                for pt in selectPixelList:
                    dsquared =(pt.x - _awayFromPoint.x)**2+(pt.y - _awayFromPoint.y)**2+(pt.z - _awayFromPoint.z)**2
                    if dsquared>maxDsquared:
                        maxDsquared=dsquared
                        farthestPoint=pt
                return farthestPoint
        else: return
    
    def getPixelsInContactWithMedium(self, _cell):
        if self.verbose: print "Trying to find pts in cell",_cell.id,"that contact Medium"
        pixelList=CellBoundaryPixelList(self.boundaryPixelTrackerPlugin,_cell)
        selectPixelList=[]
        complementPixelList=[]
        for boundaryPixelTrackerData in pixelList:
            addThisPixel=False
            pt = boundaryPixelTrackerData.pixel
            pt1 = Point3D(pt.x+1, pt.y, pt.z)
            pt2 = Point3D(pt.x, pt.y+1, pt.z)
            pt3 = Point3D(pt.x-1, pt.y, pt.z)
            pt4 = Point3D(pt.x, pt.y-1, pt.z)
            if pt.x==0:            pt3 = Point3D(self.dim.x-1, pt.y, pt.z)
            if pt.x==self.dim.x-1: pt1 = Point3D(0, pt.y, pt.z)
            if pt.y==0:            pt4 = Point3D(pt.x, self.dim.y-1, pt.z)
            if pt.y==self.dim.y-1: pt2 = Point3D(pt.x, 0, pt.z)
            cell1 = self.cellFieldG.get(pt1)
            cell2 = self.cellFieldG.get(pt2)
            cell3 = self.cellFieldG.get(pt3)
            cell4 = self.cellFieldG.get(pt4)
            for cell in [cell1, cell2, cell3, cell4]:
                if not cell:
                    selectPixelList.append(pt)
                    addThisPixel=True
            if not addThisPixel:
                complementPixelList.append(pt)
        return selectPixelList, complementPixelList
            
    def printSteppableInfo(self):
        if self.useSpecialApicalDomain:
            print "Using special apical domains to simulate apicobasal polarity"
            print "Nucleate complementary domains for each of the following pairs:"
            for typeSet in self.polarizedCompartmentsList:
                print "\t apical:",typeList[typeSet['apical']],"basolateral:",typeList[typeSet['basal']]
        if self.useSpecialJunctionDomain:
            contactStringList=[typeList[x] for x in self.nucleateWhenContactingCellTypeList]
            nucleateFromStringList=[typeList[x] for x in self.nucleateFromCellTypeList]
            print "Using special junction domains to simulate concentrated adhesion"
            print "Nucleate",typeList[TJ],"from cells of type",nucleateFromStringList
            print "\t if cell contacts any cells of type(s)",contactStringList
            print "\t with probability=",round(self.probNucleate,3),"per cell per mcs"
            print "\t up to",self.maxTJperCell,typeList[TJ],"per cell."
            print "Revert to",nucleateFromStringList,"from",typeList[TJ]
            print "\t if cell does NOT contact any cells of type(s)",contactStringList
            print "\t with probability=",round(self.probRevert,3),"per cell per mcs"
            print "Destabilize",typeList[TJ]
            print "\t if",typeList[TJ],"does NOT contact its own cluster or "
            print "\t if it contacts another",typeList[TJ],"in same cluster"
            print "\t with probability=",round(self.probDestabilize,3),"per cell per mcs"

    def updateTJs(self,mcs):
        nConversions=0
        nReversions=0
        nDestabilizations=0
        nNucleations=0
        for cell in self.cellList:
            if cell.type in self.nucleateFromCellTypeList:
                nucleationContact = self.isBoundaryCell(cell, self.nucleateWhenContactingCellTypeList)
                # find the number of TJs already in the _cell under consideration
                nTJs = 0
                if self.verbose:
                    print "ClusterId =",cell.clusterId
                    print "   nucleationContact = ",nucleationContact
                # for some reason CC3D fails if you loop over items in compartmentList instead of the indexes
                compartmentList=self.inventory.getClusterCells(cell.clusterId)
                TJList=[]
                for i in range(compartmentList.size()):
                    if compartmentList[i].type in [TJ,TJm]:
                        nTJs+=1
                        TJList.append(compartmentList[i])
                        if self.verbose: print "\t cell.id =",compartmentList[i].id,"; type=",compartmentList[i].type
                if self.verbose: print "\t Number of TJs for this cluster =",nTJs
                #DESTABILIZE extra TJs if more than maxTJperCell
                if nTJs>self.maxTJperCell:
                    for i in range(self.maxTJperCell, TJList.size()):
                        if self.verbose: print "Too many TJs. Destabilizing TJ in clusterId=",TJList[i].clusterId
                        TJList[i].targetVolume=0
                        TJList[i].targetSurface=0
                        nDestabilizations +=1
                #DESTABILIZE TJ if no longer in contact with clusterId to which it belongs
                #DESTABILIZE one of two TJs if there is a contact between two TJs in same clusterId
                #REVERT with probRevert if TJ not in contact with celltype in revertWhenNotContactingCellTypeList 
                for tjCompartment in TJList:
                    tjNeighborList=CellNeighborListAuto(self.neighborTrackerPlugin, tjCompartment)
                    contactsClusterFlag = False
                    contactsTJinClusterFlag = False
                    preventReversionContactFlag = False
                    for neighborSurfaceData in tjNeighborList:
                        if neighborSurfaceData.neighborAddress:
                            if neighborSurfaceData.neighborAddress.clusterId==tjCompartment.clusterId:
                                contactsClusterFlag=True
                                if neighborSurfaceData.neighborAddress.type in [TJ,TJm]:
                                    if neighborSurfaceData.neighborAddress.targetVolume>0:
                                        #last step tries to make sure only one of the contacting TJs is destabilized
                                        contactsTJinClusterFlag=True
                            elif neighborSurfaceData.neighborAddress.type in self.revertWhenNotContactingCellTypeList:
                                # this only gets checked if neighbors not in same cluster
                                preventReversionContactFlag=True
                    if (not contactsClusterFlag) and random.random()<self.probDestabilize:
                        if self.verbose:
                            print "Lost cluster contact."
                            print "   Destabilizing",typeList[tjCompartment.type],"(#",tjCompartment.id,") in clusterId=",tjCompartment.clusterId
                        tjCompartment.targetVolume=0
                        tjCompartment.targetSurface=0
                        nDestabilizations +=1
                    elif (contactsTJinClusterFlag) and random.random()<self.probDestabilize:
                        if self.verbose:
                            print "TJ-TJ contact within cluster."
                            print "   Destabilizing",typeList[tjCompartment.type],"(#",tjCompartment.id,") in clusterId=",tjCompartment.clusterId
                        tjCompartment.targetVolume=0
                        tjCompartment.targetSurface=0
                        nDestabilizations +=1
                    elif (not preventReversionContactFlag) and random.random()<self.probRevert:
                        if self.verbose:
                            print "No extra-cluster contact with celltype in revertWhenNotContactingCellTypeList"
                            print "   Destabilizing",typeList[tjCompartment.type],"(#",tjCompartment.id,") in clusterId=",tjCompartment.clusterId
                            tjCompartment.targetVolume=0
                            tjCompartment.targetSurface=0
                            nDestabilizations +=1
        
                #NUCLEATE TJ from self.nucleateFromCellType
                if nucleationContact and nTJs<2:
                    if random.random()<self.probNucleate:
                        if self.nucleateTJ(cell, TJList, self.nucleateWhenContactingCellTypeList): nNucleations+=1
    
                #MATURE TJ pair (or return to immature if not paired)
                for tjCompartment in TJList:
                    fppdListSize = sum(1 for _ in FocalPointPlasticityDataList(self.focalPointPlasticityPlugin, tjCompartment))
                    if fppdListSize==0:
                        if tjCompartment.type==TJm:
                            tjCompartment.type=TJ #revert to immature if not FPP-linked
                            nReversions +=1
                    else:
                        if tjCompartment.type==TJ:
                            tjCompartment.type=TJm
                            nConversions +=1
                        for fppd in FocalPointPlasticityDataList(self.focalPointPlasticityPlugin, tjCompartment):
                            if fppd.neighborAddress.type==TJ:
                                fppd.neighborAddress.type=TJm
                                nConversions+=1
                            
                    
                    
        return [nNucleations,nDestabilizations,nConversions,nReversions]

             
    def nucleateTJ(self, _cell, _TJList, _nucleateOnContactCellTypeList):
        if self.verbose: print "Entering nucleateTJ function"
        if _cell.volume>0:
            volStrain=0
            if _cell.targetVolume: volStrain = 1.0*(_cell.volume - _cell.targetVolume)/_cell.targetVolume
            # make sure deltaTargetVolume is not bigger thatn _cell.targetVolume
            deltaTargetVolume = min(self.TJTargetVolume/(1.0 + volStrain),_cell.targetVolume)
            #if _cell.targetVolume<deltaTargetVolume:  # if _cell is not big enough, don't nucleate from it
            if len(_TJList)==0 or _TJList[0].volume==0:
                pt = self.pickSurfacePixel(_cell, _nucleateOnContactCellTypeList, False) #random point on surface
            else:
                x0otherTJ = _TJList[0].xCM/float(_TJList[0].volume)
                y0otherTJ = _TJList[0].yCM/float(_TJList[0].volume)
                z0otherTJ = _TJList[0].zCM/float(_TJList[0].volume)
                centroidOtherTJ = Point3D(int(x0otherTJ),int(y0otherTJ),int(z0otherTJ))
                # point on surface farthest from the other TJ
                pt = self.pickSurfacePixel(_cell, _nucleateOnContactCellTypeList, centroidOtherTJ)
            if pt:
                if self.verbose: print "Nucleating TJ from cell.id=",_cell.id,"of type",typeList[_cell.type],"at point=",pt
                newTJ=self.Potts.createCellG(pt)

                newTJ.type=TJ
                reassignIdFlag=self.inventory.reassignClusterId(newTJ,_cell.clusterId)
                newTJ.targetVolume = self.TJTargetVolume
                newTJ.targetSurface = self.getEquivalentCircleSurface(newTJ.targetVolume)
                #use multiplier to increase chance of successful nucleation
                newTJ.lambdaVolume = self.TJLambdaVolume*self.transientNucleationMultiplier
                newTJ.lambdaSurface = self.TJLambdaSurface
                #targetDistance = 0.15*(_cell.targetVolume+2*newTJ.targetVolume)
                    #self.focalPointPlasticityPlugin.createInternalFocalPointPlasticityLink\
                   #(_cell, newTJ, self.lambdaFPP, targetDistance, 1.5*targetDistance)
                #need to check for other TJs in same cell
                    #if len(_TJList)>0:
                    #self.focalPointPlasticityPlugin.createInternalFocalPointPlasticityLink\
                     #   (_TJList[0], newTJ, self.lambdaFPP, 2.0*targetDistance, 3.0*targetDistance)
                #need to make sure all the attributes are set; need to find a better way to do this
                CompuCell.getPyAttrib(newTJ).update(DEFAULT_CELL_ATTRIBUTE_DICT)
                return newTJ
            else:
                if self.verbose: print "No TJ nucleated; could not find appropriate pt"
                return  # can use this to assess whether TJ was actually nucleated
        if self.verbose: print "No TJ nucleated; cell.volume == 0"
        return # can use this to assess whether TJ was actually nucleated

    def checkApicalDomains(self):
        # should only be one apical and one basal compartment per clusterId; this function checks and reports
        for typeSet in self.polarizedCompartmentsList:
            print "Polarized Compartment Pair: 'apical'=",typeList[typeSet['apical']],"'basal'=",typeList[typeSet['basal']]
            for compartmentList in self.clusterList:
                nApical=0
                nBasal=0
                clusterId=0
                apicalCell=0
                basalCell=0
                for cell in CompartmentList(compartmentList):
                    clusterId=cell.clusterId
                    if cell.type==typeSet['apical']:
                        nApical +=1
                        apicalCell = cell
                    if cell.type==typeSet['basal']:
                        nBasal +=1
                        basalCell = cell
 
                if nApical + nBasal >0:
                    contactString=""
                    if apicalCell and basalCell:
                        # evaluate whether these cells are neighbors
                        cellNeighborList=CellNeighborListAuto(self.neighborTrackerPlugin,apicalCell)
                        touchesBasalFlag = False
                        contactString ="NOT IN CONTACT WITH ONE ANOTHER"
                        for neighborSurfaceData in cellNeighborList:
                            if neighborSurfaceData.neighborAddress and neighborSurfaceData.neighborAddress.id==basalCell.id:
                                touchesBasalFlag=True
                        if touchesBasalFlag: contactString="contacting"
                    print "\tclusterId",clusterId,"[nApical,nBasal]=",[nApical,nBasal],"-",contactString


    def updateApicalDomains(self):
        # loop over all cells that have a type that appears somewhere in the polarizedCompartmentsList
        # for each, check if its cluster contains all the other types in its specific entry in polarizedCompartmentsList
        # if so, go on; if not, nucleate the missing compartment
        nNucleationsDict = [{'nApical':0,'nBasal':0} for entry in self.polarizedCompartmentsList]
        for cell in self.cellList:
            if cell.type in [type for typeSet in self.polarizedCompartmentsList for type in typeSet.values()]:
                #print "Cell with type in polarizedCompartmentsList: id=",cell.id,"type=",typeList[cell.type]
                compartmentList=self.inventory.getClusterCells(cell.clusterId)
                for typeSetIndex,typeSet in enumerate(self.polarizedCompartmentsList):
                    if cell.type in typeSet.values():
                        polarizationFlag=False
                        nType=0
                        for i in range(compartmentList.size()):
                            if compartmentList[i].type in typeSet.values() and compartmentList[i].type != cell.type:
                                polarizationFlag=True
                            if compartmentList[i].type == cell.type: nType +=1
                        if nType>1: print "WARNING: ClusterId",cell.clusterId,"has",nType,"compartments of type",typeList[cell.type]
                        #print "\t appropriate polarization typePair is",typeSet.values()
                        if polarizationFlag: pass  #print "\t cell is already polarized"
                        else:
                            mediumContactingPixels, otherSurfacePixels = self.getPixelsInContactWithMedium(cell)
                            #print "\t number of surface pixels in contact with medium =",len(mediumContactingPixels)
                            #print "\t number of surface pixels NOT in contact with medium =",len(otherSurfacePixels)
                            if cell.type == typeSet['apical']:
                                # create a compartment of the appropriate 'basal' type
                                if len(otherSurfacePixels)>0 and otherSurfacePixels[0]:
                                    newCell=self.Potts.createCellG(otherSurfacePixels[0])
                                        #for pt in otherSurfacePixels[1::]:
                                        #if pt:
                                    #    self.cellFieldG.set(pt,newCell)
                                    newCell.type=typeSet['basal']
                                    nNucleationsDict[typeSetIndex]['nBasal']+=1
                                    reassignIdFlag=self.inventory.reassignClusterId(newCell,cell.clusterId)
                                    newCell.targetVolume += -1.2*len(mediumContactingPixels) #0.5*cell.targetVolume
                                    cell.targetVolume =      1.2*len(mediumContactingPixels)   #0.5*cell.targetVolume
                                    newCell.targetSurface = 0.7*newCell.targetVolume
                                    cell.targetSurface = cell.targetVolume
                                    newCell.lambdaVolume = cell.lambdaVolume
                                    newCell.lambdaSurface = cell.lambdaSurface
                                    CompuCell.getPyAttrib(newCell).update(CompuCell.getPyAttrib(cell))
                                else: print "WARNING: Cell of type",typeList[cell.type],"has no basal partner, but no otherSurfacePixels for nucleation"
                            elif cell.type == typeSet['basal']:
                                # create a compartment of the appropriate 'apical' type
                                if len(mediumContactingPixels)>0 and mediumContactingPixels[0]:
                                    newCell=self.Potts.createCellG(mediumContactingPixels[0])
                                        #for pt in mediumContactingPixels[1::]:
                                        #if pt:
                                    #    self.cellFieldG.set(pt,newCell)
                                    newCell.type=typeSet['apical']
                                    nNucleationsDict[typeSetIndex]['nApical']+=1
                                    reassignIdFlag=self.inventory.reassignClusterId(newCell,cell.clusterId)
                                    newCell.targetVolume = 1.2*len(mediumContactingPixels)    #0.5*cell.targetVolume
                                    cell.targetVolume +=  -1.2*len(mediumContactingPixels)
                                    newCell.targetSurface = newCell.targetVolume
                                    cell.targetSurface = 0.7*cell.targetVolume
                                    newCell.lambdaVolume = cell.lambdaVolume
                                    newCell.lambdaSurface = cell.lambdaSurface
                                    CompuCell.getPyAttrib(newCell).update(CompuCell.getPyAttrib(cell))
                                else: print "WARNING: Cell of type",typeList[cell.type],"has no apical partner, but no mediumContactingPixels for nucleation"

        return nNucleationsDict

    

    def start(self): pass
            
    def step(self,mcs):
        t0 = time.clock()
        if mcs >= self.startMCS:
            print "\n***** ManageEpithelialPolarization (step) ***********************"
            self.printSteppableInfo()
            if self.useSpecialApicalDomain:
                nNucleationsDict = self.updateApicalDomains()
                print "----- Current Apicobasal Polarization Status -----"
                for typeSetIndex,typeSet in enumerate(self.polarizedCompartmentsList):
                    print "Number of new nucleations:\t",typeList[typeSet['apical']],"=",nNucleationsDict[typeSetIndex]['nApical'],"\t",typeList[typeSet['basal']],"=",nNucleationsDict[typeSetIndex]['nBasal']
                self.checkApicalDomains()
            if self.useSpecialJunctionDomain:
                nNucleations,nDestabilizations,nConversions,nReversions = self.updateTJs(mcs)
                nucleateFromStringList=[typeList[x] for x in self.nucleateFromCellTypeList]
                print "----- Current TJ Status -----"
                print "    Number of new nucleations=",nNucleations
                print "    Number of destabilized",typeList[TJ],"or",typeList[TJm],"=",nDestabilizations
                print "    Number of",typeList[TJ],"to",typeList[TJm],"conversions =",nConversions
                print "    Number of",typeList[TJm],"to",typeList[TJ],"reversions =",nReversions
                #self.setPurseStringPlasticity(_lambdaMAX=self.lambdaMAX, _halflife=self.halflife)
            print "\t\t\tTime Taken=", time.clock()-t0,"\n"


#*************************************************************************************************************
class SimulatePullByKillingOffCellsOnBorder(SteppableBasePy):   # simulates pulling on a patch by randomly
    def __init__(self,_simulator,_frequency=1):                 # killing off cells on a specified edge of the lattice
        SteppableBasePy.__init__(self,_simulator,_frequency)
        self.Potts = self.simulator.getPotts()
        self.cellFieldG = self.Potts.getCellFieldG()
        self.edgesToPullOn = ["Left","Right"]
        self.startPullingMCS = 4000
        self.killRate = 0.05 # fraction of edge length/mcs
        self.maxKill = 60 # multiples of cell edge length

    def start(self): pass
    
    def step(self,mcs):
        global MITOSIS_VOLUME_LIST
        if mcs > self.startPullingMCS and mcs < self.startPullingMCS + 1.0*self.maxKill/self.killRate:
            # this section (executed once) should prevent any further mitoses
            if mcs <= self.startPullingMCS+self.frequency:
                for i in range(len(MITOSIS_VOLUME_LIST)):
                    MITOSIS_VOLUME_LIST[i]=self.dim.x*self.dim.y
            print "SimulatePullByKillingOffCellsOnBorder step"
            if "Left" in self.edgesToPullOn:
                for i in range(int(self.killRate*self.frequency*self.dim.y)):
                    # pick a pixel at random along this edge (not part of the wall)
                    y = random.randint(0,self.dim.y)
                    pt = Point3D(2,y,0)
                    #cellToRemove=self.cellFieldG.get(pt)
                    cellToDecrement=self.cellFieldG.get(pt)
                    if cellToDecrement and cellToDecrement.type != WALL:
                        print "Decrementing target volume of cell",cellToDecrement.id,"of type",typeList[cellToDecrement.type]
                        cellToDecrement.targetVolume -=1
                        #cellToRemove.lambdaVolume=2*cellToRemove.lambdaVolume
                        cellToDecrement.targetSurface -=1
                        #cellToRemove.lambdaSurface=2*cellToRemove.lambdaSurface
            if "Right" in self.edgesToPullOn:
                for i in range(int(self.killRate*self.frequency*self.dim.y)):
                    # pick a pixel at random along this edge (not part of the wall)
                    y = random.randint(0,self.dim.y)
                    pt = Point3D(int(self.dim.x-2),y,0)
                    #cellToRemove=self.cellFieldG.get(pt)
                    cellToDecrement=self.cellFieldG.get(pt)
                    if cellToDecrement and cellToDecrement.type != WALL:
                        print "Decrementing target volume of cell",cellToDecrement.id,"of type",typeList[cellToDecrement.type]
                        cellToDecrement.targetVolume -=1
                        #cellToRemove.lambdaVolume=2*cellToRemove.lambdaVolume
                        cellToDecrement.targetSurface -=1
                        #cellToRemove.lambdaSurface=2*cellToRemove.lambdaSurface



#*************************************************************************************************************
